package app

import (
	"errors"
	"net/http"
	"testing"
	"time"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestGetCommunityBan(t *testing.T) {
	c.Convey("test GetCommunityBan handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		userID := "678"
		requester := "12345"
		timeout := backend.CommunityBan{
			StartTSUnixNano: time.Unix(1490732724, 0).UnixNano(),
			ModUserID:       "999",
			Reason:          "no reason",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/bans/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
				"user_id":      userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has view moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_community_moderation": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("with an existing timeout", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(timeout, true, nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						apitest.Expected = map[string]interface{}{
							"is_banned":   true,
							"start":       "2017-03-28T20:25:24Z",
							"mod_user_id": timeout.ModUserID,
							"reason":      timeout.Reason,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("without an existing timeout", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						apitest.Expected = map[string]interface{}{
							"is_banned":   false,
							"start":       "1970-01-01T00:00:00Z",
							"mod_user_id": "",
							"reason":      "",
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with error response from get timeout", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have view moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_community_moderation": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestListCommunityBans(t *testing.T) {
	c.Convey("test ListCommunityBans handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		requester := "12345"
		bans := []backend.CommunityBan{
			backend.CommunityBan{
				StartTSUnixNano: time.Unix(1490732724, 0).UnixNano(),
				ModUserID:       "999",
				Reason:          "no reason",
				UserID:          "5656",
			},
			backend.CommunityBan{
				StartTSUnixNano: time.Unix(1491722724, 0).UnixNano(),
				ModUserID:       "1010",
				Reason:          "some reason",
				UserID:          "987",
			},
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/bans/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has view moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_community_moderation": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("with existing bans", func() {
						backenderMock.On("ListCommunityBans", mock.Anything, communityID, mock.Anything, mock.Anything).
							Return(bans, nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						apitest.Expected = map[string]interface{}{
							"cursor": "1491722724000000000",
							"bans": []interface{}{
								map[string]interface{}{
									"start":       "2017-03-28T20:25:24Z",
									"mod_user_id": "999",
									"reason":      "no reason",
									"user_id":     "5656",
								},
								map[string]interface{}{
									"start":       "2017-04-09T07:25:24Z",
									"mod_user_id": "1010",
									"reason":      "some reason",
									"user_id":     "987",
								},
							},
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with error response from list bans", func() {
						backenderMock.On("ListCommunityBans", mock.Anything, communityID, mock.Anything, mock.Anything).
							Return(nil, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have view moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_community_moderation": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestAddCommunityBan(t *testing.T) {
	c.Convey("test AddCommunityBan handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		userID := "678"
		requester := "12345"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/bans/add",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id":   communityID,
				"target_user_id": userID,
				"reason":         "something",
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
					OwnerUserID: ownerUserID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"moderate_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"target_id":    userID,
						},
					}).Return(nil).Once()

					c.Convey("if the target exists and is not staff", func() {
						backenderMock.On("GetSiteUser", mock.Anything, userID).
							Return(backend.SiteUser{IsStaff: false}, true, nil).Once()

						c.Convey("if the target is not a mod", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(false, nil).Once()

							c.Convey("if the target is not banned already", func() {
								backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
									Return(backend.CommunityBan{}, false, nil).Once()

								c.Convey("if the target is not timed out", func() {
									backenderMock.On("GetCommunityTimeout", mock.Anything, communityID, userID).
										Return(backend.CommunityTimeout{}, false, nil).Once()

									c.Convey("if the call to add a ban succeeds", func() {
										backenderMock.On("AddCommunityBan", mock.Anything, mock.Anything).
											Return(nil).Once()

										c.Convey("if the user is currently in the community", func() {
											backenderMock.On("GetChannelCommunities", mock.Anything, userID).
												Return([]string{communityID}, nil).Once()

											c.Convey("if the call to unset succeeds", func() {
												backenderMock.On("SetChannelCommunities", mock.Anything, userID, []string{}).
													Return(nil).Once()
												backenderMock.On("AddCommunityModerationLog", mock.Anything, mock.Anything).Return(nil).Once()

												c.Convey("if the user has recently broadcast to the community", func() {
													backenderMock.On("ChannelRecentlyBroadcastToCommunity", mock.Anything, userID, communityID).
														Return(true, nil).Once()

													c.Convey("if the call to publish the action succeeds", func() {
														backenderMock.On("PublishCommunityModerationAction", mock.Anything, communityID, requester, userID, api.ModActionTypeBan, "something").
															Return(nil).Once()

														apitest.ExpectedStatus = http.StatusOK
														testutils.RunTest(t, s, apitest)
														time.Sleep(50 * time.Millisecond)
														backenderMock.AssertExpectations(t)
													})

													c.Convey("if the call to publish the action errors", func() {
														backenderMock.On("PublishCommunityModerationAction", mock.Anything, communityID, requester, userID, api.ModActionTypeBan, "something").
															Return(errors.New("error")).Once()

														apitest.ExpectedStatus = http.StatusOK
														testutils.RunTest(t, s, apitest)
														backenderMock.AssertExpectations(t)
													})
												})

												c.Convey("if the user has not recently broadcast to the community", func() {
													backenderMock.On("ChannelRecentlyBroadcastToCommunity", mock.Anything, userID, communityID).
														Return(false, nil).Once()

													apitest.ExpectedStatus = http.StatusOK
													testutils.RunTest(t, s, apitest)
													backenderMock.AssertExpectations(t)
												})

												c.Convey("if the call to recent broadcast errors", func() {
													backenderMock.On("ChannelRecentlyBroadcastToCommunity", mock.Anything, userID, communityID).
														Return(false, errors.New("error")).Once()

													apitest.ExpectedStatus = http.StatusOK
													testutils.RunTest(t, s, apitest)
													backenderMock.AssertExpectations(t)
												})
											})

											c.Convey("if the call to unset fails", func() {
												backenderMock.On("SetChannelCommunities", mock.Anything, userID, []string{}).
													Return(errors.New("error")).Once()

												apitest.ExpectedStatus = http.StatusInternalServerError
												testutils.RunTest(t, s, apitest)
												backenderMock.AssertExpectations(t)
											})
										})
									})

									c.Convey("if the call to add a ban errors", func() {
										backenderMock.On("AddCommunityBan", mock.Anything, mock.Anything).
											Return(errors.New("error")).Once()

										apitest.ExpectedStatus = http.StatusInternalServerError
										testutils.RunTest(t, s, apitest)
										backenderMock.AssertExpectations(t)
									})
								})
							})

							c.Convey("if the user is banned already", func() {
								backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
									Return(backend.CommunityBan{}, true, nil).Once()

								apitest.ExpectedStatus = http.StatusOK
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})

							c.Convey("if the call to check bans errors", func() {
								backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
									Return(backend.CommunityBan{}, false, errors.New("error")).Once()

								apitest.ExpectedStatus = http.StatusInternalServerError
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})
						})

						c.Convey("if the target is a mod", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(true, nil).Once()

							apitest.ExpectedStatus = http.StatusForbidden
							apitest.Expected = map[string]interface{}{
								"status": float64(http.StatusForbidden),
								"error":  api.ErrCodeTargetUserMod,
							}
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to check mod errors", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(false, errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the target exists and is staff", func() {
						backenderMock.On("GetSiteUser", mock.Anything, userID).
							Return(backend.SiteUser{IsStaff: true}, true, nil).Once()

						apitest.ExpectedStatus = http.StatusForbidden
						apitest.Expected = map[string]interface{}{
							"status": float64(http.StatusForbidden),
							"error":  api.ErrCodeTargetUserStaff,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the target doesn't exist", func() {
						backenderMock.On("GetSiteUser", mock.Anything, userID).
							Return(backend.SiteUser{}, false, nil).Once()

						apitest.ExpectedStatus = http.StatusBadRequest
						apitest.Expected = map[string]interface{}{
							"status": float64(http.StatusBadRequest),
							"error":  api.ErrCodeTargetUserNotFound,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the call to get site user errors", func() {
						backenderMock.On("GetSiteUser", mock.Anything, userID).
							Return(backend.SiteUser{}, false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"moderate_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"target_id":    userID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestRemoveCommunityBan(t *testing.T) {
	c.Convey("test RemoveCommunityBan handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		userID := "678"
		requester := "12345"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/bans/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id":   communityID,
				"target_user_id": userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
					OwnerUserID: ownerUserID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"moderate_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"target_id":    userID,
						},
					}).Return(nil).Once()

					c.Convey("if the target is banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, true, nil).Once()

						c.Convey("if the call to remove a ban succeeds", func() {
							backenderMock.On("RemoveCommunityBan", mock.Anything, communityID, userID).
								Return(nil).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to remove a ban errors", func() {
							backenderMock.On("RemoveCommunityBan", mock.Anything, communityID, userID).
								Return(errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the target is not banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the call to check ban errors", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"moderate_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
							"target_id":    userID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
