package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"

	"golang.org/x/net/context"
)

const (
	defaultFollowersLimit = 20
	maxFollowersLimit     = 100
)

func (h *handlers) ListCommunityFollowers(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListCommunityFollowersRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListCommunityFollowersParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	// specify limit
	limit := defaultFollowersLimit
	if params.Limit > 0 && params.Limit <= maxFollowersLimit {
		limit = params.Limit
	} else if params.Limit > maxFollowersLimit {
		limit = maxFollowersLimit
	}

	followers, newCursor, err := h.Backend.ListCommunityFollowers(ctx, params.CommunityID, params.Cursor, limit)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	total, err := h.Backend.CountCommunityFollowers(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := []api.CommunityFollower{}
	for _, follower := range followers {
		resp = append(resp, api.CommunityFollower{
			CommunityID: params.CommunityID,
			UserID:      follower.UserID,
			CreatedAt:   follower.CreatedAt.UTC(),
		})
	}

	gojiplus.ServeJSON(rw, req, &api.ListCommunityFollowersResponse{
		Followers: resp,
		Cursor:    newCursor,
		Total:     total,
	})
}

func validateListCommunityFollowersParams(params api.ListCommunityFollowersRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Limit < 0 {
		return errors.New("limit must not be negative")
	}
	return nil
}

func (h *handlers) CountCommunityFollowers(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.CountCommunityFollowersRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateCountCommunityFollowersParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	count, err := h.Backend.CountCommunityFollowers(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.CountCommunityFollowersResponse{
		Followers: count,
	})
}

func validateCountCommunityFollowersParams(params api.CountCommunityFollowersRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) GetCommunityFollower(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityFollowerRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityFollowerParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetSiteUser(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusNotFound)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	follower, exists, err := h.Backend.GetCommunityFollower(ctx, params.CommunityID, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	var resp *api.CommunityFollower
	if exists {
		resp = &api.CommunityFollower{
			CommunityID: params.CommunityID,
			UserID:      follower.UserID,
			CreatedAt:   follower.CreatedAt.UTC(),
		}
	}

	gojiplus.ServeJSON(rw, req, &api.GetCommunityFollowerResponse{
		Follower: resp,
	})
}

func validateGetCommunityFollowerParams(params api.GetCommunityFollowerRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	return nil
}
