package app

import (
	"net/http"
	"testing"
	"time"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
)

func TestListCommunityFollowers(t *testing.T) {
	c.Convey("test ListCommunityFollowers handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		communityID := "asdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/followers/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community exists and is not tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, true, nil).Once()

			backenderMock.On("ListCommunityFollowers", mock.Anything, communityID, mock.Anything, mock.Anything).
				Return([]backend.CommunityFollower{
					backend.CommunityFollower{
						CommunityID: communityID,
						UserID:      "123",
						CreatedAt:   time.Unix(111, 0).UTC(),
					},
					backend.CommunityFollower{
						CommunityID: communityID,
						UserID:      "456",
						CreatedAt:   time.Unix(222, 0).UTC(),
					},
					backend.CommunityFollower{
						CommunityID: communityID,
						UserID:      "789",
						CreatedAt:   time.Unix(333, 0).UTC(),
					},
				}, "101010", nil).Once()

			backenderMock.On("CountCommunityFollowers", mock.Anything, communityID).
				Return(3, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"cursor": "101010",
				"total":  float64(3),
				"followers": []interface{}{
					map[string]interface{}{
						"community_id": communityID,
						"user_id":      "123",
						"created_at":   "1970-01-01T00:01:51Z",
					},
					map[string]interface{}{
						"community_id": communityID,
						"user_id":      "456",
						"created_at":   "1970-01-01T00:03:42Z",
					},
					map[string]interface{}{
						"community_id": communityID,
						"user_id":      "789",
						"created_at":   "1970-01-01T00:05:33Z",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestCountCommunityFollowers(t *testing.T) {
	c.Convey("test CountCommunityFollowers handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		communityID := "asdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/followers/count",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community exists and is not tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, true, nil).Once()

			backenderMock.On("CountCommunityFollowers", mock.Anything, communityID).
				Return(100, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"followers": float64(100),
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestGetCommunityFollower(t *testing.T) {
	c.Convey("test GetCommunityFollower handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		communityID := "asdf"
		userID := "123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/followers/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
				"user_id":      userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if user does not exist", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeRequestingUserNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if user exists", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, true, nil).Once()

			c.Convey("if community is tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{
						TOSBanned: true,
					}, true, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityTOSBanned,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community does not exist", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityIDNotFound,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community exists and is not tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, true, nil).Once()

				c.Convey("if the user is a follower", func() {
					backenderMock.On("GetCommunityFollower", mock.Anything, communityID, userID).
						Return(backend.CommunityFollower{
							CommunityID: communityID,
							UserID:      userID,
							CreatedAt:   time.Unix(333, 0).UTC(),
						}, true, nil).Once()

					apitest.ExpectedStatus = http.StatusOK
					apitest.Expected = map[string]interface{}{
						"follower": map[string]interface{}{
							"community_id": communityID,
							"user_id":      userID,
							"created_at":   "1970-01-01T00:05:33Z",
						},
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})
		})
	})
}
