package app

import (
	"bytes"
	"encoding/base64"
	"errors"
	"fmt"
	"image"
	_ "image/jpeg"
	_ "image/png"
	"net/http"
	"strings"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"

	"github.com/pborman/uuid"
	"golang.org/x/net/context"
)

const (
	widthAvatar  = 600
	heightAvatar = 800
	widthBanner  = 1200
	heightBanner = 180

	widthAvatarResponse, heightAvatarResponse = 185, 258
	widthBannerResponse, heightBannerResponse = 1200, 180

	maxSizeAvatar = 1000000 // 1 MB
	maxSizeBanner = 3000000 // 3 MB

	noneImageName = "none"

	defaultAvatarImageURL = "https://static-cdn.jtvnw.net/twitch-community-images-production/defaults/avatar.png"
	defaultBannerImageURL = "https://static-cdn.jtvnw.net/twitch-community-images-production/defaults/banner.png"
)

func (h *handlers) UploadCommunityImages(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.UploadCommunityImageRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateUploadCommunityImageParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"edit_community_images": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	decodedBytes, err := base64.StdEncoding.DecodeString(params.ImageBytesBase64)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// detect image type and dimensions
	config, format, err := image.DecodeConfig(bytes.NewReader(decodedBytes))
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}
	if format != "png" && format != "jpeg" {
		serveError(rw, req, api.ErrCodeUnsupportedImageType, http.StatusBadRequest)
		return
	}

	// verify correct dimensions and size
	var widthExpected, heightExpected, maxSize int
	switch params.Type {
	case api.ImageTypeAvatar:
		widthExpected, heightExpected, maxSize = widthAvatar, heightAvatar, maxSizeAvatar
	case api.ImageTypeBanner:
		widthExpected, heightExpected, maxSize = widthBanner, heightBanner, maxSizeBanner
	}
	if config.Width != widthExpected || config.Height != heightExpected {
		serveError(rw, req, api.ErrCodeWrongImageDimensions, http.StatusBadRequest)
		return
	}
	if len(decodedBytes) > maxSize {
		serveError(rw, req, api.ErrCodeImageTooLarge, http.StatusBadRequest)
		return
	}

	imageID := uuid.New()
	err = h.Backend.UploadCommunityImage(ctx, params.CommunityID, imageID, format, decodedBytes)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	imageName := fmt.Sprintf("%s.%s", imageID, format)

	updateParams := backend.UpdateCommunityParams{}
	switch params.Type {
	case api.ImageTypeAvatar:
		community.AvatarImageName = imageName
		updateParams.AvatarImageName = &imageName
	case api.ImageTypeBanner:
		community.BannerImageName = imageName
		updateParams.BannerImageName = &imageName
	}

	err = h.Backend.UpdateCommunity(ctx, params.CommunityID, updateParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// update community search index
	err = h.indexCommunity(ctx, community)
	if err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusCreated)
}

func validateUploadCommunityImageParams(params api.UploadCommunityImageRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Type != api.ImageTypeAvatar && params.Type != api.ImageTypeBanner {
		return errors.New("unsupported image type parameter")
	}
	return nil
}

func (h *handlers) RemoveCommunityImages(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveCommunityImageRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveCommunityImageParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"edit_community_images": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	updateParams := backend.UpdateCommunityParams{}
	switch params.Type {
	case api.ImageTypeAvatar:
		community.AvatarImageName = noneImageName
		updateParams.AvatarImageName = &community.AvatarImageName
	case api.ImageTypeBanner:
		community.BannerImageName = noneImageName
		updateParams.BannerImageName = &community.BannerImageName
	}

	err = h.Backend.UpdateCommunity(ctx, params.CommunityID, updateParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// update community search index
	err = h.indexCommunity(ctx, community)
	if err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusCreated)
}

func validateRemoveCommunityImageParams(params api.RemoveCommunityImageRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Type != api.ImageTypeAvatar && params.Type != api.ImageTypeBanner {
		return errors.New("unsupported image type parameter")
	}

	return nil
}

func (h *handlers) communityImageURL(communityID, imageName, imageType string) string {
	if imageName == noneImageName {
		switch imageType {
		case api.ImageTypeAvatar:
			return defaultAvatarImageURL
		case api.ImageTypeBanner:
			return defaultBannerImageURL
		default:
			return ""
		}
	}

	var width, height int
	switch imageType {
	case api.ImageTypeAvatar:
		width, height = widthAvatarResponse, heightAvatarResponse
	case api.ImageTypeBanner:
		width, height = widthBannerResponse, heightBannerResponse
	default:
		return ""
	}

	parts := strings.SplitN(imageName, ".", 2)
	imageID := parts[0]
	imageFiletype := parts[1]

	if h.Conf.Environment == "production" {
		return fmt.Sprintf("https://static-cdn.jtvnw.net/community-images/%s/%s-%dx%d.%s", communityID, imageID, width, height, imageFiletype)
	}
	return fmt.Sprintf("https://twitch-community-images-staging.s3.amazonaws.com/%s/%s", communityID, imageName)
}
