package app

import (
	"bytes"
	"encoding/base64"
	"errors"
	"image"
	"image/color"
	"image/gif"
	"image/jpeg"
	"image/png"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
)

func TestUploadCommunityImagesAvatar(t *testing.T) {
	c.Convey("test UploadCommunityImages handler for avatars", t, func() {

		createImage := func(w, h int) image.Image {
			i := image.NewRGBA(image.Rectangle{image.Point{0, 0}, image.Point{w, h}})
			i.Set(2, 3, color.RGBA{255, 0, 0, 255})
			return i
		}

		pngEncode := func(i image.Image) []byte {
			buf := bytes.NewBuffer(nil)
			png.Encode(buf, i)
			return buf.Bytes()
		}

		jpegEncode := func(i image.Image) []byte {
			buf := bytes.NewBuffer(nil)
			jpeg.Encode(buf, i, nil)
			return buf.Bytes()
		}

		gifEncode := func(i image.Image) []byte {
			buf := bytes.NewBuffer(nil)
			gif.Encode(buf, i, nil)
			return buf.Bytes()
		}

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"

		bytesPng := pngEncode(createImage(widthAvatar, heightAvatar))
		bytesJpeg := jpegEncode(createImage(widthAvatar, heightAvatar))
		bytesGif := gifEncode(createImage(widthAvatar, heightAvatar))
		bytesNotImg := []byte{1, 2, 3, 4, 5, 6, 1, 2, 3, 4, 5, 6}
		bytesPngWrongSize := pngEncode(createImage(123, 456))

		strPng := base64.StdEncoding.EncodeToString(bytesPng)
		strJpeg := base64.StdEncoding.EncodeToString(bytesJpeg)
		strGif := base64.StdEncoding.EncodeToString(bytesGif)
		strNotImg := base64.StdEncoding.EncodeToString(bytesNotImg)
		strPngWrongSize := base64.StdEncoding.EncodeToString(bytesPngWrongSize)

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			OwnerUserID:         "1234",
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/images/upload",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
				"type":         api.ImageTypeAvatar,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{}, nil)

				c.Convey("if the user has edit images permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_images": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("with valid png image", func() {
						apitest.BodyJSON["image_bytes_base_64"] = strPng

						backenderMock.On("UploadCommunityImage", mock.Anything, communityID, mock.Anything, "png", bytesPng).
							Return(nil).Once()
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
							Return(nil).Once()
						backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusCreated
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with valid jpeg image", func() {
						apitest.BodyJSON["image_bytes_base_64"] = strJpeg

						backenderMock.On("UploadCommunityImage", mock.Anything, communityID, mock.Anything, "jpeg", bytesJpeg).
							Return(nil).Once()
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
							Return(nil).Once()
						backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusCreated
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with non image bytes", func() {
						apitest.BodyJSON["image_bytes_base_64"] = strNotImg

						apitest.ExpectedStatus = http.StatusBadRequest
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with wrong size image", func() {
						apitest.BodyJSON["image_bytes_base_64"] = strPngWrongSize

						apitest.ExpectedStatus = http.StatusBadRequest
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with not base64 encoded string", func() {
						apitest.BodyJSON["image_bytes_base_64"] = "asdf asdf asdf "

						apitest.ExpectedStatus = http.StatusBadRequest
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("with wrong filetype", func() {
						apitest.BodyJSON["image_bytes_base_64"] = strGif

						apitest.ExpectedStatus = http.StatusBadRequest
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have edit images permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_images": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestRemoveCommunityImagesAvatar(t *testing.T) {
	c.Convey("test RemoveCommunityImages handler for avatars", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			OwnerUserID:         "1234",
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/images/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
				"type":         api.ImageTypeAvatar,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{}, nil)

				c.Convey("if the user has edit images permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_images": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
						Return(nil).Once()
					backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
						Return(nil).Once()

					apitest.ExpectedStatus = http.StatusCreated
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("if the user does not have edit images permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_images": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
