package app

import (
	"errors"
	"fmt"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

const (
	maxModLogFilterLength = 64
	modLogDefaultPageSize = 50
	maxModLogLimit        = 50
)

func (h *handlers) GetCommunityModerationActions(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityModerationActionsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityModerationActionsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if params.Limit == 0 {
		params.Limit = modLogDefaultPageSize
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"view_community_moderation": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// specify cursor
	cursor := 0
	if params.Cursor != "" {
		cursor, err = api.CursorToInt(params.Cursor)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
			return
		}
	}

	// If there's a filter, we need to get a list of possible user IDs.
	// Algolia only gives us up to 1000 results, so hopefully that's enough.
	userIDs := []string{}
	if params.Filter != "" {
		userIDs, err = h.Backend.SearchUsernames(ctx, params.Filter)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	modLogs, modLogCount, err := h.Backend.GetCommunityModLogs(ctx, community.CommunityID, params.Filter, userIDs, int64(cursor), params.Limit)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	nextCursor := ""
	nextPagePos := cursor + params.Limit
	if int64(nextPagePos) < modLogCount {
		nextCursor = api.IntToCursorString(nextPagePos)
	}

	// So we don't get a "null" in the output JSON if there's no results
	outActions := []api.CommunityModerationAction{}
	for _, action := range modLogs {
		outActions = append(outActions, api.CommunityModerationAction(action))
	}

	resp := &api.GetCommunityModerationActionsResponse{
		Cursor:  nextCursor,
		Total:   modLogCount,
		Actions: outActions,
	}

	gojiplus.ServeJSON(rw, req, resp)
}

func validateGetCommunityModerationActionsParams(params api.GetCommunityModerationActionsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Limit < 0 || params.Limit > 50 {
		return errors.New(fmt.Sprintf("limit must be greater than zero and less than %d", maxModLogLimit))
	}
	if params.Filter != "" {
		if len(params.Filter) > maxModLogFilterLength {
			return errors.New(fmt.Sprintf("filter must be less than %d characters", maxModLogFilterLength))
		}
	}

	return nil
}
