package app

import (
	"errors"
	"net/http"
	"strconv"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

// GetCommunityMod returns whether or not a specific user is listed as a
// moderator of a community.
// NOTE: This method is used in cartman authorization functions, so must not
// require authentication.
func (h *handlers) GetCommunityMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	isMod, err := h.Backend.IsCommunityMod(ctx, params.CommunityID, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.GetCommunityModResponse{
		IsMod: isMod,
	})
}

func validateGetCommunityModParams(params api.GetCommunityModRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}
	return nil
}

// ListCommunityMods returns a list of all mods for a community.
func (h *handlers) ListCommunityMods(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListCommunityModsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListCommunityModsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	mods, err := h.Backend.ListCommunityMods(ctx, community.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}
	resp := &api.ListCommunityModsResponse{Mods: mods}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListCommunityModsParams(params api.ListCommunityModsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) AddCommunityMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddCommunityModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddCommunityModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"edit_community_moderators": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// Err if target is banned in channel
	_, exists, err = h.Backend.GetCommunityBan(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeTargetUserBanned, http.StatusBadRequest)
		return
	}

	// Return early if target is already a mod
	isMod, err := h.Backend.IsCommunityMod(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if isMod {
		rw.WriteHeader(http.StatusOK)
		return
	}

	err = h.Backend.AddCommunityMod(ctx, params.CommunityID, params.TargetUserID)
	if err == backend.ErrTargetUserAlreadyMod {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	} else if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateAddCommunityModParams(params api.AddCommunityModRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	return nil
}

func (h *handlers) RemoveCommunityMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveCommunityModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveCommunityModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"edit_community_moderators": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// Return early if target is already not a mod
	isMod, err := h.Backend.IsCommunityMod(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !isMod {
		rw.WriteHeader(http.StatusOK)
		return
	}

	err = h.Backend.RemoveCommunityMod(ctx, params.CommunityID, params.TargetUserID)
	if err == backend.ErrTargetUserNotMod {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	} else if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateRemoveCommunityModParams(params api.RemoveCommunityModRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	return nil
}
