package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestGetCommunityMod(t *testing.T) {
	c.Convey("test GetCommunityMod handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		userID := "678"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/mods/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
				"user_id":      userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if the user is a mod", func() {
			backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
				Return(true, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"is_mod": true,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the user is not a mod", func() {
			backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
				Return(false, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"is_mod": false,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the call to check mod errors", func() {
			backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
				Return(false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestListCommunityMods(t *testing.T) {
	c.Convey("test ListCommunityMods handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		mods := []string{"999", "1010"}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/mods/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
				}, true, nil).Once()

			c.Convey("with existing mods", func() {
				backenderMock.On("ListCommunityMods", mock.Anything, communityID, mock.Anything, mock.Anything).
					Return(mods, nil).Once()

				apitest.ExpectedStatus = http.StatusOK
				apitest.Expected = map[string]interface{}{
					"mods": []interface{}{
						"999", "1010",
					},
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response from list mods", func() {
				backenderMock.On("ListCommunityMods", mock.Anything, communityID, mock.Anything, mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestAddCommunityMod(t *testing.T) {
	c.Convey("test AddCommunityMod handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		userID := "678"
		requester := "12345"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/mods/add",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id":   communityID,
				"target_user_id": userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
					OwnerUserID: ownerUserID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_moderators": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("if the target is not banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, nil).Once()

						c.Convey("if the target is not already a mod", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(false, nil).Once()

							c.Convey("if the call to add a mod succeeds", func() {
								backenderMock.On("AddCommunityMod", mock.Anything, communityID, userID).
									Return(nil).Once()

								apitest.ExpectedStatus = http.StatusOK
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})

							c.Convey("if the call to add a mod errors", func() {
								backenderMock.On("AddCommunityMod", mock.Anything, communityID, userID).
									Return(errors.New("error")).Once()

								apitest.ExpectedStatus = http.StatusInternalServerError
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})

							c.Convey("if the call to add a mod errors with err target already mod", func() {
								backenderMock.On("AddCommunityMod", mock.Anything, communityID, userID).
									Return(backend.ErrTargetUserAlreadyMod).Once()

								apitest.ExpectedStatus = http.StatusOK
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})
						})

						c.Convey("if the target is already a mod", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(false, nil).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to check is mod errors", func() {
							backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
								Return(false, errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the target is banned", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, true, nil).Once()

						apitest.ExpectedStatus = http.StatusBadRequest
						apitest.Expected = map[string]interface{}{
							"status": float64(http.StatusBadRequest),
							"error":  api.ErrCodeTargetUserBanned,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the call to check bans errors", func() {
						backenderMock.On("GetCommunityBan", mock.Anything, communityID, userID).
							Return(backend.CommunityBan{}, false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_moderators": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
func TestRemoveCommunityMod(t *testing.T) {
	c.Convey("test RemoveCommunityMod handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		userID := "678"
		requester := "12345"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/mods/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id":   communityID,
				"target_user_id": userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
					OwnerUserID: ownerUserID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_moderators": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("if the target is a mod", func() {
						backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
							Return(true, nil).Once()

						c.Convey("if the call to remove a mod succeeds", func() {
							backenderMock.On("RemoveCommunityMod", mock.Anything, communityID, userID).
								Return(nil).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to remove a mod errors", func() {
							backenderMock.On("RemoveCommunityMod", mock.Anything, communityID, userID).
								Return(errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to remove a mod errors with err not mod", func() {
							backenderMock.On("RemoveCommunityMod", mock.Anything, communityID, userID).
								Return(backend.ErrTargetUserNotMod).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the target is not already a mod", func() {
						backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
							Return(false, nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})

					c.Convey("if the is mod check errors", func() {
						backenderMock.On("IsCommunityMod", mock.Anything, communityID, userID).
							Return(false, errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have moderation permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community_moderators": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
