package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"

	"golang.org/x/net/context"
)

func (h *handlers) GetCommunityPermissions(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityPermissionsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityPermissionsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	userID := token.GetSubject()

	if community.OwnerUserID == userID {
		// owner can do everything
		gojiplus.ServeJSON(rw, req, api.GetCommunityPermissionsResponse{
			Ban:     true,
			Timeout: true,
			Edit:    true,
		})
		return
	}

	siteUser, exists, err := h.Backend.GetSiteUser(ctx, userID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeTargetUserNotFound, http.StatusNotFound)
		return
	} else if siteUser.IsAdmin {
		// Admin Mods can do everything, staff have no special privs
		gojiplus.ServeJSON(rw, req, api.GetCommunityPermissionsResponse{
			Ban:     true,
			Timeout: true,
			Edit:    true,
		})
		return
	}

	isMod, err := h.Backend.IsCommunityMod(ctx, params.CommunityID, userID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if isMod {
		// mod can't edit
		gojiplus.ServeJSON(rw, req, api.GetCommunityPermissionsResponse{
			Ban:     true,
			Timeout: true,
			Edit:    false,
		})
		return
	}

	gojiplus.ServeJSON(rw, req, api.GetCommunityPermissionsResponse{
		Ban:     false,
		Timeout: false,
		Edit:    false,
	})
}

func validateGetCommunityPermissionsParams(params api.GetCommunityPermissionsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) hasCommunityModPermissions(ctx context.Context, community backend.Community, userID string) (bool, error) {
	editPermissions, err := h.hasCommunityEditPermissions(ctx, community, userID)
	if err != nil {
		return false, err
	} else if editPermissions {
		return true, nil
	}

	isMod, err := h.Backend.IsCommunityMod(ctx, community.CommunityID, userID)
	if err != nil {
		return false, err
	}
	return isMod, nil
}

func (h *handlers) hasCommunityEditPermissions(ctx context.Context, community backend.Community, userID string) (bool, error) {
	if userID == "" {
		return false, nil
	}

	if community.OwnerUserID == userID {
		return true, nil
	}
	siteUser, exists, err := h.Backend.GetSiteUser(ctx, userID)
	if err != nil {
		return false, err
	} else if !exists {
		return false, nil
	} else if siteUser.IsAdmin {
		return true, nil
	}
	return false, nil
}
