package app

import (
	"errors"
	"fmt"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"

	"golang.org/x/net/context"
)

var validReportReasons = map[string]struct{}{
	api.ReasonBitsViolation:         struct{}{},
	api.ReasonCheating:              struct{}{},
	api.ReasonGore:                  struct{}{},
	api.ReasonHarassment:            struct{}{},
	api.ReasonHarm:                  struct{}{},
	api.ReasonHateSpeech:            struct{}{},
	api.ReasonImpersonation:         struct{}{},
	api.ReasonMusicConduct:          struct{}{},
	api.ReasonNongaming:             struct{}{},
	api.ReasonOffensiveUsername:     struct{}{},
	api.ReasonOther:                 struct{}{},
	api.ReasonPorn:                  struct{}{},
	api.ReasonProhibited:            struct{}{},
	api.ReasonSelfharm:              struct{}{},
	api.ReasonSocialEatingViolation: struct{}{},
	api.ReasonSpam:                  struct{}{},
	api.ReasonTosBanEvasion:         struct{}{},
	api.ReasonUnderaged:             struct{}{},
	api.ReasonCommunityTOSViolation: struct{}{},
}

func (h *handlers) ReportCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ReportCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateReportCommunityRequest(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if _, ok := validReportReasons[params.Reason]; !ok {
		gojiplus.ServeError(ctx, rw, req, errors.New("invalid reason"), http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	communityLink := fmt.Sprintf("https://www.twitch.tv/communities/%s", community.Name)
	description := fmt.Sprintf("community report (<a href=\"%s\">%s</a>) %s", communityLink, communityLink, params.Description)

	var userID string
	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		// No auth is required to file a report.
		userID = ""
	} else {
		userID = token.GetSubject()
	}

	err = h.Backend.ReportCommunity(ctx, community, userID, params.Reason, description)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateReportCommunityRequest(params api.ReportCommunityRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}
