package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestReportCommunity(t *testing.T) {
	c.Convey("test ReportCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		requester := "123"
		reason := api.ReasonHarassment

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/report",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if report reason is valid", func() {
			apitest.BodyJSON["reason"] = reason

			c.Convey("if community exists", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, true, nil).Once()

				c.Convey("if auth token is valid", func() {
					authMock.On("ParseToken", mock.Anything).
						Return(&goauthorization.AuthorizationToken{
							Claims: goauthorization.TokenClaims{
								Sub: claim.Sub{
									Sub: requester,
								},
							},
						}, nil)

					c.Convey("with successful response for report community", func() {
						backenderMock.On("ReportCommunity", mock.Anything, mock.Anything, requester, reason, mock.Anything).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if auth token is not valid", func() {
					authMock.On("ParseToken", mock.Anything).
						Return(nil, errors.New("error"))

					c.Convey("with successful response for report community", func() {
						backenderMock.On("ReportCommunity", mock.Anything, mock.Anything, "", reason, mock.Anything).
							Return(nil).Once()

						apitest.ExpectedStatus = http.StatusOK
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})
			})

			c.Convey("if community does not exist", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityIDNotFound,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for get community", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if report reason is not valid", func() {
			apitest.BodyJSON["reason"] = "invalid reason"

			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
