package app

import (
	"errors"
	"net/http"
	"strings"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/markdown"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

func (h *handlers) GetCommunitySettings(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunitySettingsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunitySettingsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	resp := api.GetCommunitySettingsResponse{
		CommunityID:         community.CommunityID,
		Name:                community.Name,
		DisplayName:         community.DisplayName,
		OwnerUserID:         community.OwnerUserID,
		ShortDescription:    community.ShortDescription,
		LongDescription:     community.LongDescription,
		LongDescriptionHTML: community.LongDescriptionHTML,
		Rules:               community.Rules,
		RulesHTML:           community.RulesHTML,
		Language:            community.Language,
	}

	// NOTE: Email is PII, should only be exposed to the leader of the community
	// since it'll generally be their email.
	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		// Noop, no auth is required
	} else {
		err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
			"view_private_details": goauthorization.CapabilityClaim{
				"community_id": community.CommunityID,
			},
		})

		// Errors can be ignored, if the capability isn't present we just aren't
		// adding the email to the response.
		if err == nil && community.Email != noneString {
			resp.Email = community.Email
		}
	}

	if community.BannerImageName != "" && community.BannerImageName != noneImageName {
		resp.BannerImageURL = h.communityImageURL(community.CommunityID, community.BannerImageName, api.ImageTypeBanner)
	}
	if community.AvatarImageName != "" && community.AvatarImageName != noneImageName {
		resp.AvatarImageURL = h.communityImageURL(community.CommunityID, community.AvatarImageName, api.ImageTypeAvatar)
	}

	gojiplus.ServeJSON(rw, req, &resp)
}

func (h *handlers) InternalGetCommunitySettings(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunitySettingsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunitySettingsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	resp := api.GetCommunitySettingsResponse{
		CommunityID:         community.CommunityID,
		Name:                community.Name,
		DisplayName:         community.DisplayName,
		OwnerUserID:         community.OwnerUserID,
		ShortDescription:    community.ShortDescription,
		LongDescription:     community.LongDescription,
		LongDescriptionHTML: community.LongDescriptionHTML,
		Rules:               community.Rules,
		RulesHTML:           community.RulesHTML,
	}
	if community.Email != noneString {
		resp.Email = community.Email
	}
	if community.Language != noneString {
		resp.Language = community.Language
	}

	if community.BannerImageName != "" && community.BannerImageName != noneImageName {
		resp.BannerImageURL = h.communityImageURL(community.CommunityID, community.BannerImageName, api.ImageTypeBanner)
	}
	if community.AvatarImageName != "" && community.AvatarImageName != noneImageName {
		resp.AvatarImageURL = h.communityImageURL(community.CommunityID, community.AvatarImageName, api.ImageTypeAvatar)
	}

	gojiplus.ServeJSON(rw, req, &resp)
}

func validateGetCommunitySettingsParams(params api.GetCommunitySettingsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) SetCommunitySettings(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.SetCommunitySettingsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateSetCommunitySettingsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if params.DisplayName != nil && (len(*params.DisplayName) > maxDisplayNameLen || len(*params.DisplayName) < minDisplayNameLen) {
		serveError(rw, req, api.ErrCodeDisplayNameInvalid, http.StatusBadRequest)
		return
	}
	if params.LongDescription != nil && len(*params.LongDescription) > maxLongDescriptionLen {
		serveError(rw, req, api.ErrCodeLongDescriptionTooLong, http.StatusBadRequest)
		return
	}
	if params.ShortDescription != nil && len(*params.ShortDescription) > maxShortDescriptionLen {
		serveError(rw, req, api.ErrCodeShortDescriptionTooLong, http.StatusBadRequest)
		return
	}
	if params.Rules != nil && len(*params.Rules) > maxRulesLen {
		serveError(rw, req, api.ErrCodeRulesTooLong, http.StatusBadRequest)
		return
	}
	if params.Language != nil && *params.Language != "" && !isValidLanguage(*params.Language) {
		serveError(rw, req, api.ErrCodeInvalidLanguage, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"edit_community": goauthorization.CapabilityClaim{
			"community_id": community.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	if params.DisplayName != nil {
		community.DisplayName = *params.DisplayName
	}
	if params.ShortDescription != nil {
		community.ShortDescription = *params.ShortDescription
	}
	if params.LongDescription != nil {
		community.LongDescription = *params.LongDescription
		community.LongDescriptionHTML = markdown.ConvertMarkdown(community.LongDescription)
	}
	if params.Rules != nil {
		community.Rules = *params.Rules
		community.RulesHTML = markdown.ConvertMarkdown(community.Rules)
	}

	if params.Email != nil && *params.Email == "" {
		community.Email = noneString
	} else if params.Email != nil {
		community.Email = *params.Email
	}

	if params.Language != nil && *params.Language == "" {
		community.Language = noneString
	} else if params.Language != nil {
		community.Language = strings.ToUpper(*params.Language)
	}

	if params.OwnerUserID != nil {
		// Only admins can update community owners
		err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
			"edit_community_owner": goauthorization.CapabilityClaim{
				"community_id": community.CommunityID,
			},
		})
		if err != nil {
			serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
			return
		}

		_, exists, err = h.Backend.GetSiteUser(ctx, *params.OwnerUserID)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if !exists {
			serveError(rw, req, api.ErrCodeTargetUserNotFound, http.StatusNotFound)
			return
		}

		community.OwnerUserID = *params.OwnerUserID
	}

	err = h.Backend.UpdateCommunity(ctx, params.CommunityID, backend.UpdateCommunityParams{
		DisplayName:         &community.DisplayName,
		ShortDescription:    &community.ShortDescription,
		LongDescription:     &community.LongDescription,
		LongDescriptionHTML: &community.LongDescriptionHTML,
		Rules:               &community.Rules,
		RulesHTML:           &community.RulesHTML,
		Email:               &community.Email,
		Language:            &community.Language,
		OwnerUserID:         &community.OwnerUserID,
	})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// update community search index
	err = h.indexCommunity(ctx, community)
	if err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusOK)
}

func validateSetCommunitySettingsParams(params api.SetCommunitySettingsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.DisplayName != nil {
		if err := validateCommunityDisplayName(*params.DisplayName); err != nil {
			return err
		}
	}
	if params.ShortDescription != nil && *params.ShortDescription == "" {
		return errors.New("Info must not be empty")
	}
	if params.LongDescription != nil && *params.LongDescription == "" {
		return errors.New("description must not be empty")
	}
	if params.Rules != nil && *params.Rules == "" {
		return errors.New("rules must not be empty")
	}

	return nil
}
