package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestGetCommunitySettings(t *testing.T) {
	c.Convey("test GetCommunitySettings handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		requester := "12345"

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			DisplayName:         "Display name",
			OwnerUserID:         ownerUserID,
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/settings/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has view private details permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_private_details": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					apitest.ExpectedStatus = http.StatusOK
					apitest.Expected = map[string]interface{}{
						"avatar_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/fdsa-4321.png",
						"banner_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/asdf-1234.png",
						"community_id":          "asdfasdf",
						"email":                 "a@b.com",
						"language":              "EN",
						"long_description":      "long",
						"long_description_html": "<br>long",
						"name":                  "name",
						"display_name":          "Display name",
						"owner_user_id":         "555",
						"rules":                 "rules",
						"rules_html":            "<br>rules",
						"short_description":     "short",
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("if the user does not have view private details permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"view_private_details": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusOK
					apitest.Expected = map[string]interface{}{
						"avatar_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/fdsa-4321.png",
						"banner_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/asdf-1234.png",
						"community_id":          "asdfasdf",
						"email":                 "",
						"language":              "EN",
						"long_description":      "long",
						"long_description_html": "<br>long",
						"name":                  "name",
						"display_name":          "Display name",
						"owner_user_id":         "555",
						"rules":                 "rules",
						"rules_html":            "<br>rules",
						"short_description":     "short",
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusOK
				apitest.Expected = map[string]interface{}{
					"avatar_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/fdsa-4321.png",
					"banner_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/asdf-1234.png",
					"community_id":          "asdfasdf",
					"email":                 "",
					"language":              "EN",
					"long_description":      "long",
					"long_description_html": "<br>long",
					"name":                  "name",
					"display_name":          "Display name",
					"owner_user_id":         "555",
					"rules":                 "rules",
					"rules_html":            "<br>rules",
					"short_description":     "short",
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestInternalGetCommunitySettings(t *testing.T) {
	c.Convey("test InternalGetCommunitySettings handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			DisplayName:         "Display name",
			OwnerUserID:         ownerUserID,
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/settings/internal_get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			apitest.ExpectedStatus = http.StatusOK
			apitest.Expected = map[string]interface{}{
				"avatar_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/fdsa-4321.png",
				"banner_image_url":      "https://twitch-community-images-staging.s3.amazonaws.com/asdfasdf/asdf-1234.png",
				"community_id":          "asdfasdf",
				"email":                 "a@b.com",
				"language":              "EN",
				"long_description":      "long",
				"long_description_html": "<br>long",
				"name":                  "name",
				"display_name":          "Display name",
				"owner_user_id":         "555",
				"rules":                 "rules",
				"rules_html":            "<br>rules",
				"short_description":     "short",
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community is tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					TOSBanned: true,
				}, true, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityTOSBanned,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestSetCommunitySettings(t *testing.T) {
	c.Convey("test SetCommunitySettings handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		requester := "12345"

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			DisplayName:         "Display name",
			OwnerUserID:         ownerUserID,
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/settings/set",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has edit community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("if the call to update community succeeds", func() {
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
							Return(nil).Once()

						c.Convey("if the call to update the index succeeds", func() {
							backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
								Return(nil).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the call to update the index fails", func() {
							backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
								Return(errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusOK
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the call to update community fails", func() {
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
							Return(errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have edit community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestSetCommunitySettingsChangeOwner(t *testing.T) {
	c.Convey("test SetCommunitySettings handler with changing owner", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		ownerUserID := "555"
		newOwnerUserID := "666"
		requester := "12345"

		community := backend.Community{
			CommunityID:         communityID,
			Name:                "name",
			DisplayName:         "Display name",
			OwnerUserID:         ownerUserID,
			ShortDescription:    "short",
			LongDescription:     "long",
			LongDescriptionHTML: "<br>long",
			Rules:               "rules",
			RulesHTML:           "<br>rules",
			Language:            "EN",
			Email:               "a@b.com",
			BannerImageName:     "asdf-1234.png",
			AvatarImageName:     "fdsa-4321.png",
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/settings/set",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id":  communityID,
				"owner_user_id": newOwnerUserID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists and isn't tos banned", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(community, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has edit community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("if the user has edit community owner permissions", func() {
						authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
							"edit_community_owner": goauthorization.CapabilityClaim{
								"community_id": communityID,
							},
						}).Return(nil).Once()

						c.Convey("if the target exists", func() {
							backenderMock.On("GetSiteUser", mock.Anything, newOwnerUserID).
								Return(backend.SiteUser{}, true, nil).Once()

							c.Convey("if the call to update community succeeds", func() {
								backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
									Return(nil).Once()

								c.Convey("if the call to update the index succeeds", func() {
									backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
										Return(nil).Once()

									apitest.ExpectedStatus = http.StatusOK
									testutils.RunTest(t, s, apitest)
									backenderMock.AssertExpectations(t)
								})

								c.Convey("if the call to update the index fails", func() {
									backenderMock.On("IndexCommunity", mock.Anything, mock.Anything).
										Return(errors.New("error")).Once()

									apitest.ExpectedStatus = http.StatusOK
									testutils.RunTest(t, s, apitest)
									backenderMock.AssertExpectations(t)
								})
							})

							c.Convey("if the call to update community fails", func() {
								backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
									Return(errors.New("error")).Once()

								apitest.ExpectedStatus = http.StatusInternalServerError
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})
						})

						c.Convey("if the target doesn't exist", func() {
							backenderMock.On("GetSiteUser", mock.Anything, newOwnerUserID).
								Return(backend.SiteUser{}, false, nil).Once()

							apitest.ExpectedStatus = http.StatusNotFound
							apitest.Expected = map[string]interface{}{
								"status": float64(http.StatusNotFound),
								"error":  api.ErrCodeTargetUserNotFound,
							}
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})

						c.Convey("if the site user call fails", func() {
							backenderMock.On("GetSiteUser", mock.Anything, newOwnerUserID).
								Return(backend.SiteUser{}, false, errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("if the user does not have edit community owner permissions", func() {
						authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
							"edit_community_owner": goauthorization.CapabilityClaim{
								"community_id": communityID,
							},
						}).Return(errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusForbidden
						apitest.Expected = map[string]interface{}{
							"status": float64(http.StatusForbidden),
							"error":  api.ErrCodeRequestingUserNotPermitted,
						}
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have edit community permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"edit_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
