package app

import (
	"errors"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

const defaultTimeoutDurationSecs = 2 * 60 * 60  // 2 hours
const maxTimeoutDurationSecs = 7 * 24 * 60 * 60 // 1 week

func (h *handlers) GetCommunityTimeout(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetCommunityTimeoutRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetCommunityTimeoutParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"view_community_moderation": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	timeout, exists, err := h.Backend.GetCommunityTimeout(ctx, params.CommunityID, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.GetCommunityTimeoutResponse{
		IsTimedOut: exists,
		Start:      time.Unix(0, timeout.StartTSUnixNano).UTC(),
		Expiration: timeout.Expires.UTC(),
		ModUserID:  timeout.ModUserID,
		Reason:     timeout.Reason,
	})
}

func validateGetCommunityTimeoutParams(params api.GetCommunityTimeoutRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}
	return nil
}

func (h *handlers) ListCommunityTimeouts(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListCommunityTimeoutsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListCommunityTimeoutsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"view_community_moderation": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// specify cursor
	var cursor int64
	if params.Cursor == "" {
		cursor = 0
	} else {
		parsed, err := strconv.ParseInt(params.Cursor, 10, 64)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, errors.New("invalid cursor"), http.StatusBadRequest)
			return
		}
		cursor = parsed
	}

	// specify limit
	limit := params.Limit
	if limit <= 0 {
		limit = defaultLimit
	} else if limit > maxLimit {
		limit = maxLimit
	}

	Timeouts, err := h.Backend.ListCommunityTimeouts(ctx, params.CommunityID, cursor, limit)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := &api.ListCommunityTimeoutsResponse{
		Timeouts: []api.CommunityTimeout{},
	}
	for _, timeout := range Timeouts {
		resp.Timeouts = append(resp.Timeouts, api.CommunityTimeout{
			UserID:     timeout.UserID,
			Start:      time.Unix(0, timeout.StartTSUnixNano).UTC(),
			Expiration: timeout.Expires.UTC(),
			ModUserID:  timeout.ModUserID,
			Reason:     timeout.Reason,
		})
		resp.Cursor = strconv.FormatInt(timeout.StartTSUnixNano, 10)
	}

	gojiplus.ServeJSON(rw, req, resp)
}

func validateListCommunityTimeoutsParams(params api.ListCommunityTimeoutsRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if params.Cursor != "" {
		if cursor, err := strconv.ParseInt(params.Cursor, 10, 64); err != nil || cursor < 0 {
			return errors.New("invalid cursor")
		}
	}

	return nil
}

func (h *handlers) AddCommunityTimeout(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddCommunityTimeoutRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddCommunityTimeoutParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	reason := params.Reason
	if params.Reason == "" {
		reason = defaultBanReason
	}

	duration := params.Duration
	if duration == 0 {
		duration = defaultTimeoutDurationSecs
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"moderate_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
			"target_id":    params.TargetUserID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	userID := token.GetSubject()

	if params.TargetUserID == community.OwnerUserID {
		serveError(rw, req, api.ErrCodeTargetUserOwner, http.StatusForbidden)
		return
	}

	if userID == params.TargetUserID {
		serveError(rw, req, api.ErrCodeTargetUserSelf, http.StatusBadRequest)
		return
	}

	// business logic forbidding adding Timeouts
	targetUser, exists, err := h.Backend.GetSiteUser(ctx, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeTargetUserNotFound, http.StatusBadRequest)
		return
	}

	if targetUser.IsStaff {
		serveError(rw, req, api.ErrCodeTargetUserStaff, http.StatusForbidden)
		return
	}

	// noop if target is already banned in channel
	_, exists, err = h.Backend.GetCommunityBan(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	}

	startTime := time.Now()
	endTime := time.Now().Add(time.Duration(duration) * time.Second)
	err = h.Backend.AddCommunityTimeout(ctx, backend.CommunityModerationAction{
		CommunityID:  params.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Type:         "timeout",
		Reason:       reason,
		StartTime:    startTime,
		EndTime:      &endTime,
	})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	currentCommunities, err := h.Backend.GetChannelCommunities(ctx, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if contains(currentCommunities, params.CommunityID) {
		// currently broadcasting to this community - unset them
		err = h.Backend.SetChannelCommunities(ctx, params.TargetUserID, remove(currentCommunities, params.CommunityID))
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	if err := h.Backend.AddCommunityModerationLog(ctx, backend.CommunityModerationAction{
		CommunityID:  community.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Type:         "timeout",
		StartTime:    startTime,
		EndTime:      &endTime,
		Reason:       params.Reason,
	}); err != nil {
		logx.Error(ctx, err)
	}

	recentlyBroadcast, err := h.Backend.ChannelRecentlyBroadcastToCommunity(ctx, params.TargetUserID, params.CommunityID)
	if err != nil {
		logx.Error(ctx, err)
	} else if recentlyBroadcast {
		err = h.Backend.PublishCommunityModerationAction(ctx, params.CommunityID, userID, params.TargetUserID, api.ModActionTypeTimeout, reason)
		if err != nil {
			logx.Error(ctx, err)
		}
	}

	rw.WriteHeader(http.StatusOK)
}

func validateAddCommunityTimeoutParams(params api.AddCommunityTimeoutRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	if params.Reason == "" {
		params.Reason = defaultBanReason
	}
	if params.Duration < 0 || params.Duration > maxTimeoutDurationSecs {
		return errors.New("invalid duration")
	}
	return nil
}

func (h *handlers) RemoveCommunityTimeout(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveCommunityTimeoutRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveCommunityTimeoutParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"moderate_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
			"target_id":    params.TargetUserID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// noop if target is not already Timed Out
	_, exists, err = h.Backend.GetCommunityTimeout(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		// noop
		rw.WriteHeader(http.StatusOK)
		return
	}

	err = h.Backend.RemoveCommunityTimeout(ctx, params.CommunityID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	startTime := time.Now()
	userID := token.GetSubject()
	if err := h.Backend.AddCommunityModerationLog(ctx, backend.CommunityModerationAction{
		CommunityID:  community.CommunityID,
		ModUserID:    userID,
		TargetUserID: params.TargetUserID,
		Type:         "untimeout",
		StartTime:    startTime,
	}); err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusOK)
}

func validateRemoveCommunityTimeoutParams(params api.RemoveCommunityTimeoutRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}

	return nil
}
