package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"

	"golang.org/x/net/context"
)

func (h *handlers) TOSBanCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.TOSBanCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateTOSBanCommunityRequest(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	token, err := h.Authorization.ParseToken(req)
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusUnauthorized)
		return
	}
	err = h.Authorization.Validate(token, goauthorization.CapabilityClaims{
		"tosban_community": goauthorization.CapabilityClaim{
			"community_id": params.CommunityID,
		},
	})
	if err != nil {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}
	userID := token.GetSubject()

	community.TOSBanned = true
	err = h.Backend.UpdateCommunity(ctx, params.CommunityID, backend.UpdateCommunityParams{
		TOSBanned:       &community.TOSBanned,
		TOSBanRequester: &userID,
	})
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// remove from search index
	err = h.Backend.UnindexCommunity(ctx, community.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// get channels currently broadcasting to community
	channelIDs, err := h.Backend.GetChannelsByCommunity(ctx, community.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	err = h.Backend.BulkUnsetChannelCommunities(ctx, channelIDs)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateTOSBanCommunityRequest(params api.TOSBanCommunityRequest) error {
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}
