package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	mockauth "code.justin.tv/chat/zuma/app/mocks/auth"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
)

func TestTOSBanCommunity(t *testing.T) {
	c.Convey("test TOSBanCommunity handler", t, func() {

		authMock := &mockauth.IDecoder{}

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend:       backenderMock,
			Authorization: authMock,
		})

		communityID := "asdfasdf"
		requester := "12345"
		channelsOnCommunity := []string{"123", "456", "789"}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/tos_ban",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{
					CommunityID: communityID,
				}, true, nil).Once()

			c.Convey("if the auth token parses", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(&goauthorization.AuthorizationToken{
						Claims: goauthorization.TokenClaims{
							Sub: claim.Sub{
								Sub: requester,
							},
						},
					}, nil)

				c.Convey("if the user has tos ban permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"tosban_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(nil).Once()

					c.Convey("with successful response for update community", func() {
						boolptr := true
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, backend.UpdateCommunityParams{
							TOSBanned:       &boolptr,
							TOSBanRequester: &requester,
						}).Return(nil).Once()

						c.Convey("with successful response for unindex community", func() {
							backenderMock.On("UnindexCommunity", mock.Anything, communityID).
								Return(nil).Once()

							c.Convey("with successul response for get channels by community", func() {
								backenderMock.On("GetChannelsByCommunity", mock.Anything, communityID).
									Return(channelsOnCommunity, nil).Once()

								c.Convey("with successful response for bulk unset", func() {
									backenderMock.On("BulkUnsetChannelCommunities", mock.Anything, channelsOnCommunity).
										Return(nil).Once()

									apitest.ExpectedStatus = http.StatusOK
									testutils.RunTest(t, s, apitest)
									backenderMock.AssertExpectations(t)
								})

								c.Convey("with error response from bulk unset", func() {
									backenderMock.On("BulkUnsetChannelCommunities", mock.Anything, channelsOnCommunity).
										Return(errors.New("error")).Once()

									apitest.ExpectedStatus = http.StatusInternalServerError
									testutils.RunTest(t, s, apitest)
									backenderMock.AssertExpectations(t)
								})
							})

							c.Convey("with error response for get channels by community", func() {
								backenderMock.On("GetChannelsByCommunity", mock.Anything, communityID).
									Return(nil, errors.New("error")).Once()

								apitest.ExpectedStatus = http.StatusInternalServerError
								testutils.RunTest(t, s, apitest)
								backenderMock.AssertExpectations(t)
							})
						})

						c.Convey("with error response for unindex community", func() {
							backenderMock.On("UnindexCommunity", mock.Anything, communityID).
								Return(errors.New("error")).Once()

							apitest.ExpectedStatus = http.StatusInternalServerError
							testutils.RunTest(t, s, apitest)
							backenderMock.AssertExpectations(t)
						})
					})

					c.Convey("with error response for update community", func() {
						backenderMock.On("UpdateCommunity", mock.Anything, communityID, mock.Anything).
							Return(errors.New("error")).Once()

						apitest.ExpectedStatus = http.StatusInternalServerError
						testutils.RunTest(t, s, apitest)
						backenderMock.AssertExpectations(t)
					})
				})

				c.Convey("if the user does not have tos ban permissions", func() {
					authMock.On("Validate", mock.Anything, goauthorization.CapabilityClaims{
						"tosban_community": goauthorization.CapabilityClaim{
							"community_id": communityID,
						},
					}).Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusForbidden
					apitest.Expected = map[string]interface{}{
						"status": float64(http.StatusForbidden),
						"error":  api.ErrCodeRequestingUserNotPermitted,
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})

			c.Convey("if the token does not parse", func() {
				authMock.On("ParseToken", mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusUnauthorized
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusUnauthorized),
					"error":  api.ErrCodeRequestingUserNotPermitted,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
