package app

import (
	"net/http"
	"sort"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"

	"golang.org/x/net/context"
)

// filterLiveCommunitiesToFeaturedCommunities takes the full list of live
// communities, removes all non-featured communities, and adds non-live featured
// communities at the end of the list (with 0 viewers and 0 channels)
func (h *handlers) filterLiveCommunitiesToFeaturedCommunities(ctx context.Context, liveCommunities []backend.LiveCommunityStats) ([]backend.LiveCommunityStats, error) {
	// get all featured communities
	featuredCommunityIDs, err := h.Backend.FeaturedCommunityIDs(ctx)
	if err != nil {
		return nil, err
	}

	// convert to map for easy access
	featuredCommunityIDsMap := map[string]struct{}{}
	for _, id := range featuredCommunityIDs {
		featuredCommunityIDsMap[id] = struct{}{}
	}

	resp := []backend.LiveCommunityStats{}

	// add live featured communities to response
	for _, lc := range liveCommunities {
		if _, ok := featuredCommunityIDsMap[lc.CommunityID]; ok {
			resp = append(resp, lc)
		}
		delete(featuredCommunityIDsMap, lc.CommunityID)
	}

	// add non-live featured communities to response
	// we need them to be ordered consistently, so we sort them by community ID
	nonLiveFeaturedCommunities := []string{}
	for id := range featuredCommunityIDsMap {
		nonLiveFeaturedCommunities = append(nonLiveFeaturedCommunities, id)
	}
	sort.Strings(nonLiveFeaturedCommunities)
	for _, id := range nonLiveFeaturedCommunities {
		resp = append(resp, backend.LiveCommunityStats{
			CommunityID: id,
			Viewers:     0,
			Channels:    0,
		})
	}

	return resp, nil
}

func (h *handlers) ListFeaturedCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListFeaturedCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// these results are sorted by ID
	communityIDs, err := h.Backend.FeaturedCommunityIDs(ctx)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	communities, err := h.Backend.BulkGetCommunities(ctx, communityIDs)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// generate results in the same order as the original communityIDs
	results := []api.Community{}
	for _, communityID := range communityIDs {
		if community, ok := communities[communityID]; ok {
			results = append(results, h.apiCommunityFromBackendCommunity(community))
		}
	}
	gojiplus.ServeJSON(rw, req, &api.ListFeaturedCommunitiesResponse{
		FeaturedCommunities: results,
	})
}

func (h *handlers) apiCommunityFromBackendCommunity(community backend.Community) api.Community {
	resp := api.Community{
		CommunityID:      community.CommunityID,
		Name:             community.Name,
		DisplayName:      community.DisplayName,
		OwnerUserID:      community.OwnerUserID,
		ShortDescription: community.ShortDescription,
		LongDescription:  community.LongDescription,
		Rules:            community.Rules,
	}
	if community.Email != noneString {
		resp.Email = community.Email
	}
	if community.Language != noneString {
		resp.Language = community.Language
	}

	if community.BannerImageName != "" && community.BannerImageName != noneImageName {
		resp.BannerImageURL = h.communityImageURL(community.CommunityID, community.BannerImageName, api.ImageTypeBanner)
	}
	if community.AvatarImageName != "" && community.AvatarImageName != noneImageName {
		resp.AvatarImageURL = h.communityImageURL(community.CommunityID, community.AvatarImageName, api.ImageTypeAvatar)
	}
	return resp
}

func (h *handlers) AddFeaturedCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddFeaturedCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	err = h.Backend.AddFeaturedCommunityID(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func (h *handlers) RemoveFeaturedCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveFeaturedCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	err = h.Backend.RemoveFeaturedCommunityID(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}
