package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestListFeaturedCommunities(t *testing.T) {
	c.Convey("test ListFeaturedCommunities handler", t, func() {

		var (
			testCommunityA = backend.Community{
				CommunityID:      "abc",
				Name:             "mycommunity",
				DisplayName:      "mycommunity",
				OwnerUserID:      "111",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "this@is.fake",
				BannerImageName:  "asdfasdf.png",
				AvatarImageName:  "123.png",
			}
			testCommunityB = backend.Community{
				CommunityID:      "querty",
				Name:             "blahblah",
				DisplayName:      "blahblah",
				OwnerUserID:      "222",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "rules",
				Language:         "EN",
				Email:            "none",
				BannerImageName:  "testest.png",
				AvatarImageName:  "xkcd.png",
			}
			testCommunityC = backend.Community{
				CommunityID:      "lol",
				Name:             "hellothere",
				DisplayName:      "hellothere",
				OwnerUserID:      "444",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "a@b.c",
				BannerImageName:  "lolol.png",
				AvatarImageName:  "abc.png",
			}
		)

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
			Conf: config.Config{
				Environment: "production",
			},
		})

		featuredCommunityIDs := []string{testCommunityA.CommunityID, testCommunityB.CommunityID, testCommunityC.CommunityID}
		bulkCommunitiesResp := map[string]backend.Community{
			testCommunityA.CommunityID: testCommunityA,
			testCommunityB.CommunityID: testCommunityB,
			testCommunityC.CommunityID: testCommunityC,
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/featured/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"asdf": "asdf",
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with successful response for featured community IDs", func() {
			backenderMock.On("FeaturedCommunityIDs", mock.Anything).
				Return(featuredCommunityIDs, nil).Once()

			c.Convey("with successful response for bulk get", func() {
				backenderMock.On("BulkGetCommunities", mock.Anything, featuredCommunityIDs).Return(bulkCommunitiesResp, nil).Once()

				apitest.Expected = map[string]interface{}{
					"featured_communities": []interface{}{
						map[string]interface{}{
							"avatar_image_url":  "https://static-cdn.jtvnw.net/community-images/abc/123-185x258.png",
							"banner_image_url":  "https://static-cdn.jtvnw.net/community-images/abc/asdfasdf-1200x180.png",
							"community_id":      "abc",
							"email":             "this@is.fake",
							"language":          "EN",
							"long_description":  "long long long",
							"name":              "mycommunity",
							"display_name":      "mycommunity",
							"owner_user_id":     "111",
							"rules":             "no rules",
							"short_description": "short short",
						},
						map[string]interface{}{
							"avatar_image_url":  "https://static-cdn.jtvnw.net/community-images/querty/xkcd-185x258.png",
							"banner_image_url":  "https://static-cdn.jtvnw.net/community-images/querty/testest-1200x180.png",
							"community_id":      "querty",
							"email":             "",
							"language":          "EN",
							"long_description":  "long long long",
							"name":              "blahblah",
							"display_name":      "blahblah",
							"owner_user_id":     "222",
							"rules":             "rules",
							"short_description": "short short",
						},
						map[string]interface{}{
							"avatar_image_url":  "https://static-cdn.jtvnw.net/community-images/lol/abc-185x258.png",
							"banner_image_url":  "https://static-cdn.jtvnw.net/community-images/lol/lolol-1200x180.png",
							"community_id":      "lol",
							"email":             "a@b.c",
							"language":          "EN",
							"long_description":  "long long long",
							"name":              "hellothere",
							"display_name":      "hellothere",
							"owner_user_id":     "444",
							"rules":             "no rules",
							"short_description": "short short",
						},
					},
				}

				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for bulk get", func() {
				backenderMock.On("BulkGetCommunities", mock.Anything, mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("with error response for featured community IDs", func() {
			backenderMock.On("FeaturedCommunityIDs", mock.Anything).
				Return(nil, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestAddFeaturedCommunity(t *testing.T) {
	c.Convey("test AddFeaturedCommunity handler", t, func() {

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		communityID := "asdfasdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/featured/add",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, true, nil).Once()

			c.Convey("with successful response for add featured community id", func() {
				backenderMock.On("AddFeaturedCommunityID", mock.Anything, communityID).
					Return(nil).Once()

				apitest.ExpectedStatus = http.StatusOK
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for add featured community id", func() {
				backenderMock.On("AddFeaturedCommunityID", mock.Anything, communityID).
					Return(errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestRemoveFeaturedCommunity(t *testing.T) {
	c.Convey("test RemoveFeaturedCommunity handler", t, func() {

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		communityID := "asdfasdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/featured/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if community exists", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, true, nil).Once()

			c.Convey("with successful response for remove featured community id", func() {
				backenderMock.On("RemoveFeaturedCommunityID", mock.Anything, communityID).
					Return(nil).Once()

				apitest.ExpectedStatus = http.StatusOK
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for remove featured community id", func() {
				backenderMock.On("RemoveFeaturedCommunityID", mock.Anything, communityID).
					Return(errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("if community does not exist", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeCommunityIDNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with error response for get community", func() {
			backenderMock.On("GetCommunity", mock.Anything, communityID).
				Return(backend.Community{}, false, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestFilterLiveCommunitiesToFeaturedCommunities(t *testing.T) {
	c.Convey("test filterLiveCommunitiesToFeaturedCommunities method", t, func() {
		backenderMock := &backend.MockBackender{}

		h := &handlers{
			Backend: backenderMock,
		}

		featuredCommunityIDs := []string{"abc", "123", "asdf"}
		liveCommunities := []backend.LiveCommunityStats{
			backend.LiveCommunityStats{
				CommunityID: "test",
				Viewers:     100,
				Channels:    25,
			},
			backend.LiveCommunityStats{
				CommunityID: "abc",
				Viewers:     77,
				Channels:    33,
			},
			backend.LiveCommunityStats{
				CommunityID: "testing",
				Viewers:     55,
				Channels:    12,
			},
			backend.LiveCommunityStats{
				CommunityID: "asdf",
				Viewers:     43,
				Channels:    10,
			},
			backend.LiveCommunityStats{
				CommunityID: "lolol",
				Viewers:     42,
				Channels:    12,
			},
		}
		ctx := context.Background()

		c.Convey("with successful response for featured community ids", func() {
			backenderMock.On("FeaturedCommunityIDs", mock.Anything).
				Return(featuredCommunityIDs, nil).Once()

			filtered, err := h.filterLiveCommunitiesToFeaturedCommunities(ctx, liveCommunities)
			c.So(err, c.ShouldBeNil)
			c.So(filtered, c.ShouldResemble, []backend.LiveCommunityStats{
				backend.LiveCommunityStats{
					CommunityID: "abc",
					Viewers:     77,
					Channels:    33,
				},
				backend.LiveCommunityStats{
					CommunityID: "asdf",
					Viewers:     43,
					Channels:    10,
				},
				backend.LiveCommunityStats{
					CommunityID: "123",
					Viewers:     0,
					Channels:    0,
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with an error response for featured community ids", func() {
			backenderMock.On("FeaturedCommunityIDs", mock.Anything).
				Return(nil, errors.New("error")).Once()

			_, err := h.filterLiveCommunitiesToFeaturedCommunities(ctx, liveCommunities)
			c.So(err, c.ShouldNotBeNil)
			backenderMock.AssertExpectations(t)
		})
	})
}
