package app

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/internal/models"
)

func (h *handlers) AddMember(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.AddMemberRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateAddMemberParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	c := models.Entity{params.ContainerEntity}

	// To add a member to a container, create a default container view.
	cv := models.ContainerView{
		UserID:          params.UserID,
		ContainerEntity: c,
		JoinedAt:        h.Backend.TimeNow(),
	}
	containerView, ok, err := h.Backend.CreateContainerViewIfNotExists(ctx, cv)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	if !ok {
		// User is already a member; fetch the existing container view.
		current, err := h.Backend.GetContainerView(ctx, params.UserID, c)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		} else if current == nil {
			// The container view was unable to be created nor fetched. This is
			// a race condition. Return a 5xx to suggest that the caller retries.
			err = errx.New("container view not found (race condition encountered)")
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
		containerView = *current
	}

	resp := api.AddMemberResponse{
		UserID:          containerView.UserID,
		ContainerEntity: containerView.ContainerEntity.Entity,
		JoinedAt:        containerView.JoinedAt,
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateAddMemberParams(params api.AddMemberRequest) error {
	if params.ContainerEntity.ID() == "" {
		return errx.New("container ID is invalid")
	}

	if params.UserID == "" {
		return errx.New("user ID is invalid")
	}

	return nil
}
