package app

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
)

func (h *handlers) ApproveMessage(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.ApproveMessageRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateApproveMessageParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// An approved message is published and marked as no longer requiring review.
	isPublished := true
	isPendingReview := false
	updateParams := backend.UpdateMessageParams{
		IsPublished:     &isPublished,
		IsPendingReview: &isPendingReview,
		UpdatedAt:       h.Backend.TimeNow(),
	}
	m, err := h.Backend.UpdateMessage(ctx, params.MessageID, params.ApproverID, updateParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// Asynchronously handle post-request updates.
	go h.handlePublishedMessage(context.Background(), m)

	resp := api.ApproveMessageResponse{
		Message: exportMessage(m),
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateApproveMessageParams(params api.ApproveMessageRequest) error {
	if params.MessageID == "" {
		return errx.New("message ID is invalid")
	}

	if params.ApproverID == "" {
		return errx.New("approver user ID is invalid")
	}

	return nil
}
