package app

import (
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

const (
	createContainerMaxMemberUsersLimit = 20
)

func (h *handlers) CreateContainer(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.CreateContainerRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateCreateContainerParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	c := models.Entity{params.ContainerEntity}

	// Create a container view for each member.
	for _, userID := range params.MemberUserIDs {
		cv := models.ContainerView{
			UserID:          userID,
			ContainerEntity: c,
			JoinedAt:        h.Backend.TimeNow(),
		}
		if _, _, err := h.Backend.CreateContainerViewIfNotExists(ctx, cv); err != nil {
			logx.Error(ctx, errx.Wrap(err, "creating container view for member"))
		}
	}

	// Update container after adding users so the member count is accurate.
	var containerOwner *models.Entity
	if params.ContainerOwner != nil {
		containerOwner = &models.Entity{*params.ContainerOwner}
	}
	databag := models.Databag(params.Databag)
	updateContainerParams := backend.UpdateContainerParams{
		ContainerOwner: containerOwner,
		RequireReview:  &params.RequireReview,
		Databag:        &databag,
	}
	container, err := h.Backend.UpdateContainer(ctx, c, updateContainerParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := api.CreateContainerResponse{
		Container: exportContainer(container),
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateCreateContainerParams(params api.CreateContainerRequest) error {
	if params.ContainerEntity.ID() == "" {
		return errx.New("container is invalid")
	}

	if !models.IsValidRequireReviewEnum(params.RequireReview) {
		return errx.New("require review field is invalid")
	}

	if len(params.MemberUserIDs) > createContainerMaxMemberUsersLimit {
		return errx.New(fmt.Errorf("cannot create container with more than %d initial members", createContainerMaxMemberUsersLimit))
	}

	return nil
}
