package app

import (
	"net/http"
	"strconv"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

func (h *handlers) CreateMessage(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.CreateMessageRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateCreateMessageParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	sentAt := h.Backend.TimeNow()
	messageID := h.Backend.GenerateUUID()

	var orderByKey string
	if params.OrderByKey != nil {
		orderByKey = *params.OrderByKey
	} else {
		// If order_by_key is not set, default to sent_at timestamp.
		orderByKey = strconv.FormatInt(sentAt.UnixNano(), 10)
	}

	c := models.Entity{params.ContainerEntity}
	container, err := h.Backend.GetContainer(ctx, c)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if container == nil {
		// If container does not exist, use a default container.
		container = &models.Container{
			ContainerEntity: c,
			RequireReview:   models.RequireReviewNone,
		}
	}

	isPublished := true
	isPendingReview := false
	if container.RequireReview == models.RequireReviewAll {
		isPublished = false
		isPendingReview = true
	}

	// Save the message object to the database.
	m := models.Message{
		ID:              messageID,
		ContainerEntity: container.ContainerEntity,
		ContainerType:   container.ContainerEntity.Namespace(),
		ContainerOwner:  container.Owner,
		Content:         importMessageContent(params.Content),
		Sender:          importMessageSender(params.Sender),
		SentAt:          sentAt,
		Databag:         models.Databag(params.Databag),
		OrderByKey:      orderByKey,
		IsPublished:     isPublished,
		IsPendingReview: isPendingReview,
	}
	message, err := h.Backend.CreateMessage(ctx, m)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// Asynchronously handle post-request updates if the message was published.
	if m.IsPublished {
		go h.handlePublishedMessage(context.Background(), message)
	}

	resp := api.CreateMessageResponse{
		Message: exportMessage(message),
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func (h *handlers) handlePublishedMessage(ctx context.Context, m models.Message) {
	c := m.ContainerEntity

	if !backend.ShouldStoreLastMessageInContainerView(c.Namespace()) {
		// The last message is stored on the container.
		// Update the sender's container view.
		if _, err := h.updateContainerViewAfterNewMessage(ctx, m.Sender.UserID, c, m); err != nil {
			logx.Error(ctx, errx.Wrap(err, "updating sender's container view after new message"))
		}

		// Update the last message on the container.
		updateContainerParams := backend.UpdateContainerParams{
			LastMessage: &m,
		}
		if _, err := h.Backend.UpdateContainer(ctx, c, updateContainerParams); err != nil {
			logx.Error(ctx, errx.Wrap(err, "updating container after new message"))
		}

		// Publish to the container.
		if shouldPublishToPubsub(c) {
			publishParams := backend.PublishMessageToPubsubParams{
				Message: m,
			}
			if err := h.Backend.PublishMessageToPubsub(ctx, publishParams); err != nil {
				logx.Error(ctx, errx.Wrap(err, "publishing to pubsub after new message"))
			}
		}

		return
	}

	// The last message is stored in the container view.
	memberIDs := []string{}
	listParams := backend.ListMembersParams{Limit: 100}
	members, cursor, err := h.Backend.ListMembers(ctx, c, listParams)
	if err != nil {
		logx.Error(ctx, errx.Wrap(err, "fetching members after new message"))
		return
	} else if cursor != "" {
		// Cursor should be empty; there should not be a container that
		// that fans out to more than 100 members.
		logx.Error(ctx, "non-empty cursor found when listing members", logx.Fields{"container_id": c})
	}
	for _, member := range members {
		memberIDs = append(memberIDs, member.UserID)
	}

	// Update the container view for all members (including the sender).
	for _, userID := range memberIDs {
		if _, err := h.updateContainerViewAfterNewMessage(ctx, userID, c, m); err != nil {
			logx.Error(ctx, errx.Wrap(err, "updating container view after new message"))
		}
	}

	// Publish to all members.
	if shouldPublishToPubsub(c) {
		publishParams := backend.PublishMessageToPubsubParams{
			Message: m,
			UserIDs: memberIDs,
		}
		if err := h.Backend.PublishMessageToPubsub(ctx, publishParams); err != nil {
			logx.Error(ctx, errx.Wrap(err, "publishing to pubsub after new message"))
		}
	}
}

// updateContainerViewWithNewMessage will update a container view, or create one
// if it doesn't exist, after a new message is sent.
func (h *handlers) updateContainerViewAfterNewMessage(ctx context.Context, userID string, c models.Entity, m models.Message) (models.ContainerView, error) {
	isSender := (userID == m.Sender.UserID)

	current, err := h.Backend.GetContainerView(ctx, userID, c)
	if err != nil {
		return models.ContainerView{}, errx.Wrap(err, "updating container view")
	}

	// Container view does not exist; create one.
	if current == nil {
		cv := models.ContainerView{
			UserID:             userID,
			ContainerEntity:    c,
			JoinedAt:           h.Backend.TimeNow(),
			LastMessage:        &m,
			LastSentAtUnixNano: m.SentAt.UnixNano(),
		}

		if isSender {
			cv.LastReadAt = m.SentAt
			cv.UserSendCount = 1
			cv.UserLastSentAt = m.SentAt
			cv.UnreadNoticeCount = 0
		} else {
			cv.UnreadNoticeCount = 1
		}

		containerView, ok, err := h.Backend.CreateContainerViewIfNotExists(ctx, cv)
		if err != nil {
			return models.ContainerView{}, errx.Wrap(err, "updating container view")
		} else if !ok {
			return models.ContainerView{}, errx.New("unable to create container view (race condition)")
		}
		return containerView, nil
	}

	// Container view exists; update it.
	isArchived := false
	updateParams := backend.UpdateContainerViewIfExistsParams{
		// Unarchive a container view is a new message is received.
		IsArchived:  &isArchived,
		LastMessage: &m,
	}

	// Update the LastSentAtUnixNano field for unmuted containers views.
	if !current.IsMuted {
		messageUnixNano := updateParams.LastMessage.SentAt.UnixNano()
		updateParams.LastSentAtUnixNano = &messageUnixNano
	}

	if isSender {
		updateParams.LastReadAt = &m.SentAt
		updateParams.IncrUserSendCount = 1
		updateParams.UserLastSentAt = &m.SentAt
		updateParams.ResetUnreadNoticeCount = true
	} else {
		updateParams.IncrUnreadNoticeCount = 1
	}

	containerView, ok, err := h.Backend.UpdateContainerViewIfExists(ctx, userID, c, updateParams)
	if err != nil {
		return models.ContainerView{}, errx.Wrap(err, "updating container view")
	} else if !ok {
		return models.ContainerView{}, errx.New("unable to update container view (race condition)")
	}
	return containerView, nil
}

func shouldPublishToPubsub(c models.Entity) bool {
	return c.Namespace() == models.ContainerTypeWhispers
}

func validateCreateMessageParams(params api.CreateMessageRequest) error {
	if params.ContainerEntity.ID() == "" {
		return errx.New("container is invalid")
	}

	if len(params.Content.Text) == 0 {
		return errx.New("message text is invalid")
	}

	if params.Sender.UserID == "" {
		return errx.New("message sender is invalid")
	}

	return nil
}
