package app

import (
	"bytes"
	"context"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/backend/backendfakes"
	"code.justin.tv/chat/zuma/internal/models"
	"code.justin.tv/feeds/feeds-common/entity"
)

func TestExtractMessage(t *testing.T) {
	fakeBackender := getFakeBackender("111111")
	s, _ := New(Params{Backend: fakeBackender})
	ts := httptest.NewServer(s)
	defer ts.Close()

	owner := entity.New("user", "222222")
	reqStruct := api.ExtractMessageRequest{
		SenderID:       "111111",
		MessageText:    "The quick brown fox blah blah!",
		ContainerOwner: &owner,
	}
	b, _ := json.Marshal(reqStruct)
	req, _ := http.NewRequest("POST", ts.URL+"/v1/maap/extraction/extract", bytes.NewReader(b))
	resp, _ := http.DefaultClient.Do(req)

	var respStruct api.ExtractMessageResponse
	err := json.NewDecoder(resp.Body).Decode(&respStruct)
	assert.Nil(t, err)

	expected := api.ExtractMessageResponse{
		Risk: api.MessageRisk{
			SpamLikelihood: 0.322,
			IsAutomodClean: true,
		},
		Content: api.MessageContent{
			Text: "The quick brown fox blah blah!",
			Emoticons: []api.MessageContentEmoticon{
				{ID: "3", Start: 10, End: 14},
			},
		},
		Sender: api.MessageSender{
			UserID:      "111111",
			Login:       "test_user",
			DisplayName: "Test_User",
			ChatColor:   "#AABBCC",
			Badges: []api.UserBadge{
				{ID: "staff", Version: "1"},
			},
		},
	}
	assert.Equal(t, expected, respStruct)
}

func TestEnforceMessage(t *testing.T) {
	getDefaultParams := func() (api.ExtractMessageRequest, extractMessageData) {
		params := api.ExtractMessageRequest{}

		requiredData := extractMessageData{
			automodResp: backend.AutoModResponse{
				IsClean:             true,
				ContainsBannedWords: false,
			},
			spamLikelihood: 0.322,
			senderSiteUser: backend.SiteUser{
				HasViolatedDMCA: false,
				HasViolatedTOS:  false,
				IsDeleted:       false,
				VerifiedEmail:   true,
			},
			ownerRoomProperties: &backend.RoomProperties{
				ChatRequireVerifiedAccount: false,
			},
			isSenderChannelBanned: false,
			isSenderIgnored:       false,
		}

		return params, requiredData
	}

	var (
		err          error
		params       api.ExtractMessageRequest
		requiredData extractMessageData
	)

	// OK: no enforcement error
	params, requiredData = getDefaultParams()
	err = enforceMessage(params, requiredData)
	assert.Nil(t, err)

	// Failed: spam score too high
	params, requiredData = getDefaultParams()
	requiredData.spamLikelihood = 1.0
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrLikelySpam, err)

	// Failed: has banned words
	params, requiredData = getDefaultParams()
	requiredData.automodResp.ContainsBannedWords = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrContainsBannedWord, err)

	// Failed: sender is DMCA banned
	params, requiredData = getDefaultParams()
	requiredData.senderSiteUser.HasViolatedDMCA = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderIsSuspendedOrDeleted, err)

	// Failed: sender is TOS banned
	params, requiredData = getDefaultParams()
	requiredData.senderSiteUser.HasViolatedTOS = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderIsSuspendedOrDeleted, err)

	// Failed: sender is deleted
	params, requiredData = getDefaultParams()
	requiredData.senderSiteUser.IsDeleted = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderIsSuspendedOrDeleted, err)

	// OK: sender does not have verified email and verified emails are not enforced
	params, requiredData = getDefaultParams()
	requiredData.senderSiteUser.VerifiedEmail = false
	err = enforceMessage(params, requiredData)
	assert.Nil(t, err)

	// Failed: sender does not have verified email and verified emails are enforced
	params, requiredData = getDefaultParams()
	requiredData.senderSiteUser.VerifiedEmail = false
	requiredData.ownerRoomProperties.ChatRequireVerifiedAccount = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderHasUnverifiedEmail, err)

	// Failed: sender is banned in the owner's channel
	params, requiredData = getDefaultParams()
	requiredData.isSenderChannelBanned = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderIsChannelBanned, err)

	// Failed: sender is ignored by the owner
	params, requiredData = getDefaultParams()
	requiredData.isSenderIgnored = true
	err = enforceMessage(params, requiredData)
	assert.Equal(t, api.ErrSenderIsIgnored, err)
}

func TestFetchExtractMessageDataWithContainerOwner(t *testing.T) {
	var (
		senderID         = "111111"
		containerOwnerID = "222222"
		containerOwner   = entity.New("user", containerOwnerID)
		messageText      = "hello world"

		arg2, arg3, arg4 interface{}
	)

	fakeBackender := getFakeBackender(senderID)
	h := &handlers{Backend: fakeBackender}
	extractData, err := h.fetchExtractMessageData(context.Background(), senderID, messageText, &containerOwner)
	assert.Nil(t, err)

	expected := extractMessageData{
		automodResp: backend.AutoModResponse{
			IsClean:             true,
			ContainsBannedWords: false,
		},
		spamLikelihood: 0.322,
		senderTMIUser: backend.TMIUser{
			UserID:         senderID,
			HideTurboBadge: false,
			Color:          "#AABBCC",
		},
		senderSiteUser: backend.SiteUser{
			CreatedOn:       time.Unix(0, 0),
			DisplayName:     "Test_User",
			IsAdmin:         false,
			IsGlobalMod:     false,
			IsStaff:         true,
			Login:           "test_user",
			UserID:          senderID,
			VerifiedEmail:   false,
			Email:           "test@test.com",
			HasViolatedDMCA: false,
			HasViolatedTOS:  false,
			IsDeleted:       false,
		},
		senderBadges: []models.UserBadge{
			{ID: "staff", Version: "1"},
		},
		senderEmoteSets:       []int{11, 22, 33},
		isSenderChannelBanned: false,
		isSenderIgnored:       false,
		ownerRoomProperties: &backend.RoomProperties{
			ChatRequireVerifiedAccount: false,
		},
	}
	assert.Equal(t, expected, extractData)

	assert.Equal(t, 1, fakeBackender.GetSpamConfidenceCallCount())
	_, arg2, arg3 = fakeBackender.GetSpamConfidenceArgsForCall(0)
	assert.Equal(t, senderID, arg2)
	assert.Equal(t, messageText, arg3)

	assert.Equal(t, 1, fakeBackender.GetRoomPropertiesCallCount())
	_, arg2 = fakeBackender.GetRoomPropertiesArgsForCall(0)
	assert.Equal(t, containerOwnerID, arg2)

	assert.Equal(t, 1, fakeBackender.GetSiteUserCallCount())
	_, arg2 = fakeBackender.GetSiteUserArgsForCall(0)
	assert.Equal(t, senderID, arg2)

	assert.Equal(t, 1, fakeBackender.GetTMIUserCallCount())
	_, arg2 = fakeBackender.GetTMIUserArgsForCall(0)
	assert.Equal(t, senderID, arg2)

	assert.Equal(t, 1, fakeBackender.GetUserBadgesCallCount())
	_, arg2, arg3 = fakeBackender.GetUserBadgesArgsForCall(0)
	assert.Equal(t, senderID, arg2)
	assert.Equal(t, &containerOwnerID, arg3)

	assert.Equal(t, 1, fakeBackender.AutoModCheckMessageCallCount())
	_, arg2, arg3, arg4 = fakeBackender.AutoModCheckMessageArgsForCall(0)
	assert.Equal(t, senderID, arg2)
	assert.Equal(t, containerOwnerID, arg3)
	assert.Equal(t, messageText, arg4)

	assert.Equal(t, 1, fakeBackender.GetBanCallCount())
	_, arg2, arg3 = fakeBackender.GetBanArgsForCall(0)
	assert.Equal(t, containerOwnerID, arg2)
	assert.Equal(t, senderID, arg3)

	assert.Equal(t, 1, fakeBackender.ListUserBlocksCallCount())
	_, arg2 = fakeBackender.ListUserBlocksArgsForCall(0)
	assert.Equal(t, containerOwnerID, arg2)
}

func TestFetchExtractMessageDataWithoutContainerOwner(t *testing.T) {
	var (
		senderID    = "111111"
		messageText = "hello world"

		arg2, arg3 interface{}
	)

	fakeBackender := getFakeBackender(senderID)
	h := &handlers{Backend: fakeBackender}
	extractData, err := h.fetchExtractMessageData(context.Background(), senderID, messageText, nil)
	assert.Nil(t, err)

	expected := extractMessageData{
		automodResp: backend.AutoModResponse{
			IsClean:             true,
			ContainsBannedWords: false,
		},
		spamLikelihood: 0.322,
		senderTMIUser: backend.TMIUser{
			UserID:         senderID,
			HideTurboBadge: false,
			Color:          "#AABBCC",
		},
		senderSiteUser: backend.SiteUser{
			CreatedOn:       time.Unix(0, 0),
			DisplayName:     "Test_User",
			IsAdmin:         false,
			IsGlobalMod:     false,
			IsStaff:         true,
			Login:           "test_user",
			UserID:          senderID,
			VerifiedEmail:   false,
			Email:           "test@test.com",
			HasViolatedDMCA: false,
			HasViolatedTOS:  false,
			IsDeleted:       false,
		},
		senderBadges: []models.UserBadge{
			{ID: "staff", Version: "1"},
		},
		senderEmoteSets:       []int{11, 22, 33},
		isSenderChannelBanned: false,
		isSenderIgnored:       false,
		ownerRoomProperties:   nil,
	}
	assert.Equal(t, expected, extractData)

	assert.Equal(t, 1, fakeBackender.GetSpamConfidenceCallCount())
	_, arg2, arg3 = fakeBackender.GetSpamConfidenceArgsForCall(0)
	assert.Equal(t, senderID, arg2)
	assert.Equal(t, messageText, arg3)

	assert.Zero(t, fakeBackender.GetRoomPropertiesCallCount())

	assert.Equal(t, 1, fakeBackender.GetSiteUserCallCount())
	_, arg2 = fakeBackender.GetSiteUserArgsForCall(0)
	assert.Equal(t, senderID, arg2)

	assert.Equal(t, 1, fakeBackender.GetTMIUserCallCount())
	_, arg2 = fakeBackender.GetTMIUserArgsForCall(0)
	assert.Equal(t, senderID, arg2)

	assert.Equal(t, 1, fakeBackender.GetUserBadgesCallCount())
	_, arg2, arg3 = fakeBackender.GetUserBadgesArgsForCall(0)
	assert.Equal(t, senderID, arg2)
	assert.Nil(t, arg3)

	assert.Zero(t, fakeBackender.AutoModCheckMessageCallCount())

	assert.Zero(t, fakeBackender.GetBanCallCount())

	assert.Zero(t, fakeBackender.ListUserBlocksCallCount())
}

func getFakeBackender(senderID string) *backendfakes.FakeBackender {
	var b backendfakes.FakeBackender

	b.GetSpamConfidenceReturns(0.322, nil)

	b.GetRoomPropertiesReturns(backend.RoomProperties{
		ChatRequireVerifiedAccount: false,
	}, true, nil)

	b.GetSiteUserReturns(backend.SiteUser{
		CreatedOn:       time.Unix(0, 0),
		DisplayName:     "Test_User",
		IsAdmin:         false,
		IsGlobalMod:     false,
		IsStaff:         true,
		Login:           "test_user",
		UserID:          senderID,
		VerifiedEmail:   false,
		Email:           "test@test.com",
		HasViolatedDMCA: false,
		HasViolatedTOS:  false,
		IsDeleted:       false,
	}, true, nil)

	b.GetTMIUserReturns(backend.TMIUser{
		UserID:         senderID,
		HideTurboBadge: false,
		Color:          "#AABBCC",
	}, true, nil)

	b.GetUserBadgesReturns([]models.UserBadge{{ID: "staff", Version: "1"}}, nil)

	b.AutoModCheckMessageReturns(backend.AutoModResponse{
		IsClean:             true,
		ContainsBannedWords: false,
	}, nil)

	b.GetBanReturns(backend.Ban{}, false, nil)

	b.ListUserBlocksReturns([]string{"666666"}, nil)

	b.GetEmoteSetsReturns([]int{11, 22, 33}, nil)

	b.ParseEmoticonsReturns([]models.MessageContentEmoticon{
		{ID: "3", Start: 10, End: 14},
	}, nil)

	return &b
}

func TestTruncate(t *testing.T) {
	var s string

	s = "This string is way too long and should be truncated."
	s = truncate(s, 30)
	assert.Equal(t, "This string is way too long an", s)

	s = "This string is way shorter."
	s = truncate(s, 30)
	assert.Equal(t, "This string is way shorter.", s)
}

func TestSanitize(t *testing.T) {
	var s string

	s = "I have an \x00invalid unicode characters\n\t"
	s = sanitize(s)
	assert.Equal(t, "I have an invalid unicode characters", s)

	s = "I have inv\u115F\uFFA0alid hangul characters"
	s = sanitize(s)
	assert.Equal(t, "I have invalid hangul characters", s)

	s = "   I have    excess    whitespace   "
	s = sanitize(s)
	assert.Equal(t, "I have    excess    whitespace", s)
}

func TestCensor(t *testing.T) {
	var s string
	var found []string

	// Test no match
	s, found = censor("xfoo foox xfoox", "***", []string{"foo"})
	assert.Equal(t, "xfoo foox xfoox", s)
	assert.Len(t, found, 0)

	// Test exact match
	s, found = censor("foo xfoo foox xfoox", "***", []string{"foo"})
	assert.Equal(t, "*** xfoo foox xfoox", s)
	assert.Len(t, found, 1)

	// Test prefix match
	s, found = censor("foo xfoo foox xfoox", "***", []string{"foo*"})
	assert.Equal(t, "*** xfoo *** xfoox", s)
	assert.Len(t, found, 2)

	// Test suffix match
	s, found = censor("foo xfoo foox xfoox", "***", []string{"*foo"})
	assert.Equal(t, "*** *** foox xfoox", s)
	assert.Len(t, found, 2)

	// Test inner match
	s, found = censor("foo xfoo foox xfoox", "***", []string{"*foo*"})
	assert.Equal(t, "*** *** *** ***", s)
	assert.Len(t, found, 4)

	// Test that tokens and banned words are case insensitive
	s, found = censor("foo foO fOo fOO Foo FoO FOo FOO", "***", []string{"Foo"})
	assert.Equal(t, "*** *** *** *** *** *** *** ***", s)
	assert.Len(t, found, 8)

	// Test that whitespace is preserved
	s, found = censor(" foo  foo   foo    ", "***", []string{"foo"})
	assert.Equal(t, " ***  ***   ***    ", s)
	assert.Len(t, found, 3)

	// Test multiple matches
	s, found = censor("the quick brown fox", "***", []string{"QUICK*", "fox"})
	assert.Equal(t, "the *** brown ***", s)
	assert.Len(t, found, 2)
}
