package app

import (
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
)

const (
	listContainerViewsByUserMinLimit = 1
	listContainerViewsByUserMaxLimit = 50
)

func (h *handlers) ListContainerViewsByUser(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.ListContainerViewsByUserRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateListContainerViewsByUserParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	ListContainerViewsByUserParams := backend.ListContainerViewsByUserParams{
		Limit:         params.Limit,
		Cursor:        params.Cursor,
		SortAscending: params.SortAscending,
	}
	containerViews, cursor, err := h.Backend.ListContainerViewsByUser(ctx, params.UserID, params.ContainerType, ListContainerViewsByUserParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	exportedContainerViews := []api.ContainerView{}
	for _, cv := range containerViews {
		exportedContainerViews = append(exportedContainerViews, exportContainerView(cv))
	}
	resp := api.ListContainerViewsByUserResponse{
		ContainerViews: exportedContainerViews,
		Cursor:         cursor,
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListContainerViewsByUserParams(params api.ListContainerViewsByUserRequest) error {
	if params.UserID == "" {
		return errx.New("user ID is invalid")
	}

	if params.ContainerType == "" {
		return errx.New("container type is invalid")
	}

	if params.Limit < listContainerViewsByUserMinLimit || params.Limit > listContainerViewsByUserMaxLimit {
		return errx.New(fmt.Sprintf("limit must be between %d and %d", listContainerViewsByUserMinLimit, listContainerViewsByUserMaxLimit))
	}

	return nil
}
