package app

import (
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

const (
	listContainersByOwnerMinLimit = 1
	listContainersByOwnerMaxLimit = 50
)

func (h *handlers) ListContainersByOwner(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.ListContainersByOwnerRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateListContainersByOwnerParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	listParams := backend.ListContainersByOwnerParams{
		Limit:  params.Limit,
		Cursor: params.Cursor,
	}
	containers, cursor, err := h.Backend.ListContainersByOwner(ctx, params.ContainerType, models.Entity{params.ContainerOwner}, listParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	exportedContainers := []api.Container{}
	for _, c := range containers {
		exportedContainers = append(exportedContainers, exportContainer(c))
	}
	resp := api.ListContainersByOwnerResponse{
		Containers: exportedContainers,
		Cursor:     cursor,
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListContainersByOwnerParams(params api.ListContainersByOwnerRequest) error {
	if params.ContainerType == "" {
		return errx.New("container type is invalid")
	}

	if params.ContainerOwner.ID() == "" {
		return errx.New("container owner is invalid")
	}

	if params.Limit < listContainersByOwnerMinLimit || params.Limit > listContainersByOwnerMaxLimit {
		return errx.New(fmt.Sprintf("limit must be between %d and %d", listContainersByOwnerMinLimit, listContainersByOwnerMaxLimit))
	}

	return nil
}
