package app

import (
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

const (
	listMembersMinLimit = 1
	listMembersMaxLimit = 1000
)

func (h *handlers) ListMembers(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.ListMembersRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateListMembersParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	c := models.Entity{params.ContainerEntity}
	container, err := h.Backend.GetContainer(ctx, c)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// If container doesn't exist, return 0 members.
	if container == nil {
		resp := api.ListMembersResponse{
			Members: []api.MemberUser{},
			Cursor:  "",
			Total:   0,
		}
		gojiplus.ServeJSON(rw, req, resp)
		return
	}

	listParams := backend.ListMembersParams{
		Limit:  params.Limit,
		Cursor: params.Cursor,
	}
	members, cursor, err := h.Backend.ListMembers(ctx, c, listParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	exportedMembers := []api.MemberUser{}
	for _, m := range members {
		exportedMembers = append(exportedMembers, api.MemberUser(m))
	}
	resp := api.ListMembersResponse{
		Members: exportedMembers,
		Cursor:  cursor,
		Total:   container.MemberCount,
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListMembersParams(params api.ListMembersRequest) error {
	if params.ContainerEntity.ID() == "" {
		return errx.New("container is invalid")
	}

	if params.Limit < listMembersMinLimit || params.Limit > listMembersMaxLimit {
		return errx.New(fmt.Sprintf("limit must be between %d and %d", listMembersMinLimit, listMembersMaxLimit))
	}

	return nil
}
