package app

import (
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

const (
	listMessageMinLimit = 1
	listMessageMaxLimit = 1000
)

func (h *handlers) ListMessages(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.ListMessagesRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateListMessageParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	listMessagesParams := backend.ListMessagesParams{
		Limit:                   params.Limit,
		Cursor:                  params.Cursor,
		SortAscending:           params.SortAscending,
		ReturnOnlyPendingReview: params.ReturnOnlyPendingReview,
	}
	messages, cursor, err := h.Backend.ListMessages(ctx, models.Entity{params.ContainerEntity}, listMessagesParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	exportedMessages := []api.Message{}
	for _, m := range messages {
		exportedMessages = append(exportedMessages, exportMessage(m))
	}
	resp := api.ListMessagesResponse{
		Messages: exportedMessages,
		Cursor:   cursor,
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListMessageParams(params api.ListMessagesRequest) error {
	if params.ContainerEntity.ID() == "" {
		return errx.New("container is invalid")
	}

	if params.Limit < listMessageMinLimit || params.Limit > listMessageMaxLimit {
		return errx.New(fmt.Sprintf("limit must be between %d and %d", listMessageMinLimit, listMessageMaxLimit))
	}

	return nil
}
