package app

import (
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
	"code.justin.tv/feeds/feeds-common/entity"
)

// Methods in this file convert primitives and internal, backend structs into
// external, exported structs that are part of the public interface.

// exportMessageContentEmoticons assembles a slice of exported "MessageContentEmoticon" objects.
func exportMessageContentEmoticons(emoticons []models.MessageContentEmoticon) []api.MessageContentEmoticon {
	res := []api.MessageContentEmoticon{}
	for _, e := range emoticons {
		res = append(res, api.MessageContentEmoticon(e))
	}
	return res
}

// exportUserBadges assembles a slice of exported "UserBadge" objects.
func exportUserBadges(badges []models.UserBadge) []api.UserBadge {
	res := []api.UserBadge{}
	for _, b := range badges {
		res = append(res, api.UserBadge(b))
	}
	return res
}

// exportMessageRisk assembles the "Risk" field in the extract response.
func exportMessageRisk(spamLikelihood float64, automodResp backend.AutoModResponse) api.MessageRisk {
	return api.MessageRisk{
		SpamLikelihood: spamLikelihood,
		IsAutomodClean: automodResp.IsClean,
	}
}

// exportMessageContent assembles the "Content" field in the extract response.
func exportMessageContent(mc models.MessageContent) api.MessageContent {
	return api.MessageContent{
		Text:      mc.Text,
		Emoticons: exportMessageContentEmoticons(mc.Emoticons),
	}
}

// exportMessageSender assembles the "Sender" field in the extract response.
func exportMessageSender(ms models.MessageSender) api.MessageSender {
	return api.MessageSender{
		UserID:      ms.UserID,
		Login:       ms.Login,
		DisplayName: ms.DisplayName,
		ChatColor:   ms.ChatColor,
		Badges:      exportUserBadges(ms.Badges),
	}
}

// exportMessage assembles the external "Message" object.
func exportMessage(m models.Message) api.Message {
	return api.Message{
		ID:              m.ID,
		ContainerEntity: m.ContainerEntity.Entity,
		Content:         exportMessageContent(m.Content),
		Sender:          exportMessageSender(m.Sender),
		SentAt:          m.SentAt,
		EditedAt:        m.EditedAt,
		DeletedAt:       m.DeletedAt,
		IsPublished:     m.IsPublished,
		IsPendingReview: m.IsPendingReview,
		Databag:         api.Databag(m.Databag),
	}
}

// exportContainer assembles the external "Container" object.
func exportContainer(c models.Container) api.Container {
	var owner *entity.Entity
	if c.Owner != nil {
		owner = &c.Owner.Entity
	}

	var lastMessage *api.Message
	if c.LastMessage != nil {
		m := exportMessage(*c.LastMessage)
		lastMessage = &m
	}

	return api.Container{
		ContainerEntity: c.ContainerEntity.Entity,
		Owner:           owner,
		RequireReview:   c.RequireReview,
		MemberCount:     c.MemberCount,
		LastMessage:     lastMessage,
		Databag:         api.Databag(c.Databag),
	}
}

// exportContainerView assembles the external "ContainerView" object.
func exportContainerView(cv models.ContainerView) api.ContainerView {
	var lastMessage *api.Message
	if cv.LastMessage != nil {
		m := exportMessage(*cv.LastMessage)
		lastMessage = &m
	}

	return api.ContainerView{
		UserID:            cv.UserID,
		IsArchived:        cv.IsArchived,
		IsMuted:           cv.IsMuted,
		LastReadAt:        cv.LastReadAt,
		UserSendCount:     cv.UserSendCount,
		UserLastSentAt:    cv.UserLastSentAt,
		LastMessage:       lastMessage,
		JoinedAt:          cv.JoinedAt,
		HasUnreadMessages: cv.HasUnreadMessages,
		UnreadNoticeCount: cv.UnreadNoticeCount,
	}
}
