package app

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

func (h *handlers) UpdateContainerView(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var params api.UpdateContainerViewRequest
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if err := validateUpdateContainerViewParams(params); err != nil {
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	c := models.Entity{params.ContainerEntity}

	current, err := h.Backend.GetContainerView(ctx, params.UserID, c)
	if err != nil {
		err := errx.New("container view not found")
		gojiplus.ServePublicError(ctx, rw, req, err, http.StatusNotFound)
		return
	}

	updateParams := backend.UpdateContainerViewIfExistsParams{
		IsArchived: params.IsArchived,
		IsMuted:    params.IsMuted,
		LastReadAt: params.LastReadAt,
	}

	// If archiving a container, remove the attribute.
	// If unarchiving an unmuted container, update to the last message's timestamp.
	if updateParams.IsArchived != nil {
		if *updateParams.IsArchived {
			// Negative values for LastSentAtUnixNano will remove the attribute.
			flagForRemoval := int64(-1)
			updateParams.LastSentAtUnixNano = &flagForRemoval
		} else if !current.IsMuted && current.LastMessage != nil {
			messageUnixNano := current.LastMessage.SentAt.UnixNano()
			updateParams.LastSentAtUnixNano = &messageUnixNano
		}
	}

	// If marking the container view as read, assume all unread notices have been read.
	if updateParams.LastReadAt != nil {
		updateParams.ResetUnreadNoticeCount = true
	}

	containerView, ok, err := h.Backend.UpdateContainerViewIfExists(ctx, params.UserID, c, updateParams)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !ok {
		err := errx.New("container view not found (race condition)")
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := api.UpdateContainerViewResponse{
		ContainerView: exportContainerView(containerView),
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateUpdateContainerViewParams(params api.UpdateContainerViewRequest) error {
	if params.UserID == "" {
		return errx.New("user ID is invalid")
	}

	if params.ContainerEntity.ID() == "" {
		return errx.New("container ID is invalid")
	}

	return nil
}
