package app

import (
	"errors"
	"net/http"
	"strconv"

	"code.justin.tv/chat/golibs/async"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"

	"golang.org/x/net/context"
)

func (h *handlers) ParseMessage(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ParseMessageRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateParseMessageParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	resp, err := h.doMessageParsing(ctx, params.ChannelID, params.UserID, params.Body)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, resp)
}

func validateParseMessageParams(params api.ParseMessageRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}
	if len(params.Body) == 0 {
		return errors.New("body must be present")
	}
	return nil
}

func (h *handlers) doMessageParsing(ctx context.Context, channelID, userID, messageBody string) (api.ParseMessageResponse, error) {
	var ban backend.Ban
	var isBanned bool
	var tmiRoom backend.RoomProperties
	var user backend.SiteUser
	var emoteSets []int
	var badges []models.UserBadge
	var userColor string
	fns := []func() error{
		func() error {
			// check if the user is banned in this channel
			var err error
			ban, isBanned, err = h.Backend.GetBan(ctx, channelID, userID)
			return err
		},
		func() error {
			// channel/room properties: banned words opt-out, require verified email
			var err error
			var roomExists bool
			tmiRoom, roomExists, err = h.Backend.GetRoomProperties(ctx, channelID)
			if err != nil {
				return err
			} else if !roomExists {
				return errx.New("channel does not exist", errx.Fields{
					"channel_id": channelID,
				})
			}
			return nil
		},
		func() error {
			// user properties: login, display name, email verified
			var err error
			var userFound bool
			user, userFound, err = h.Backend.GetSiteUser(ctx, userID)
			if err != nil {
				return err
			} else if !userFound {
				return errx.New("Site user not found", errx.Fields{
					"user_id": userID,
				})
			}
			return nil
		},
		func() error {
			// load emotesets
			var err error
			emoteSets, err = h.Backend.GetUserEmoteSets(ctx, userID)
			return err
		},
		func() error {
			// load user badges
			var err error
			badges, err = h.Backend.GetUserBadges(ctx, userID, &channelID)
			return err
		},
		func() error {
			// user color
			tmiUser, userFound, err := h.Backend.GetTMIUser(ctx, userID)
			if err != nil {
				return err
			} else if !userFound {
				return errx.New("TMI user not found", errx.Fields{
					"user_id": userID,
				})
			}
			userColor = tmiUser.Color
			return nil
		},
	}

	err := async.DoAsync(fns...)
	if err != nil {
		return api.ParseMessageResponse{}, err
	}

	if isBanned {
		return api.ParseMessageResponse{
			Status:      api.ParseMessageStatusUserBanned,
			BannedUntil: ban.EndTime,
		}, nil
	}

	if channelID != userID && tmiRoom.ChatRequireVerifiedAccount && !user.VerifiedEmail {
		return api.ParseMessageResponse{
			Status: api.ParseMessageStatusUserNotVerified,
		}, nil
	}

	// remove banned words
	sanitizedBody := sanitize(messageBody)
	bannedWords, err := h.allBannedWords(ctx, channelID, tmiRoom.GlobalBannedWordsOptout)
	if err != nil {
		return api.ParseMessageResponse{}, err
	}
	sanitizedBody, _ = censor(sanitizedBody, "***", bannedWords)

	// parse emoticons
	emoticons, err := h.Backend.ParseEmoticons(ctx, sanitizedBody, emoteSets)
	if err != nil {
		return api.ParseMessageResponse{}, err
	}

	// automod
	automodResp, err := h.Backend.AutoModCheckMessage(ctx, userID, channelID, sanitizedBody)
	if err != nil {
		return api.ParseMessageResponse{}, err
	}
	var status api.ParseMessageStatus
	if automodResp.IsClean {
		status = api.ParseMessageStatusOK
	} else {
		status = api.ParseMessageStatusReviewRequired
	}

	return api.ParseMessageResponse{
		Status: status,
		Message: &api.ParseMessageResult{
			Body:            sanitizedBody,
			Emoticons:       emoticonsFromParsedEmoticons(emoticons),
			UserBadges:      badgesFromRetrievedBadges(badges),
			UserColor:       userColor,
			UserLogin:       user.Login,
			UserDisplayName: user.DisplayName,
		},
	}, nil
}

func (h *handlers) allBannedWords(ctx context.Context, channelID string, optedOut bool) ([]string, error) {
	var globalBannedWords []string
	var channelBannedWords [][]string

	fns := []func() error{
		func() error {
			var err error
			globalBannedWords, err = h.Backend.GetGlobalBannedWords(ctx, optedOut)
			return err
		},
		func() error {
			var err error
			channelBannedWords, err = h.Backend.GetChannelBannedWords(ctx, channelID)
			return err
		},
	}

	err := async.DoAsync(fns...)
	if err != nil {
		return nil, err
	}

	allBannedWords := globalBannedWords
	for _, channelBannedPhrases := range channelBannedWords {
		allBannedWords = append(allBannedWords, channelBannedPhrases...)
	}

	return allBannedWords, nil
}

func emoticonsFromParsedEmoticons(parsedEmoticons []models.MessageContentEmoticon) []api.ParseMessageEmoticonRange {
	emoticons := make([]api.ParseMessageEmoticonRange, len(parsedEmoticons))
	for idx, r := range parsedEmoticons {
		emoticons[idx] = api.ParseMessageEmoticonRange{
			ParseMessageBodyRange: api.ParseMessageBodyRange{
				Begin: r.Start,
				End:   r.End,
			},
			ID: r.ID,
		}
	}
	return emoticons
}

func badgesFromRetrievedBadges(retrievedBadges []models.UserBadge) []api.ParseMessageUserBadge {
	badges := make([]api.ParseMessageUserBadge, len(retrievedBadges))
	for idx, b := range retrievedBadges {
		badges[idx] = api.ParseMessageUserBadge{
			ID:      b.ID,
			Version: b.Version,
		}
	}
	return badges
}
