package app

import (
	"net/http"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
	"code.justin.tv/chat/zuma/internal/models"
)

func TestHandlersParseMessage(t *testing.T) {
	c.Convey("ParseMessage", t, func() {
		stats, _ := statsd.NewNoopClient()
		backenderMock := &backend.MockBackender{}
		userID := "9999"
		channelID := "10001"
		messageBody := "Jebaited KAPPA OUTDATED Jebaited POGCHAMP OVERRATED Jebaited LONG HAVE WE WAITED Jebaited NOW WE JEBAITED Jebaited"
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		c.Convey("with invalid request", func() {
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/messages/parse",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": "invalid garbage",
					"user_id":    "asdfasdf",
					"body":       "",
				},
				ExpectedStatus: http.StatusBadRequest,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with a valid request", func() {
			c.Convey("when the user is not banned", func() {
				backenderMock.On("GetBan", mock.Anything, channelID, userID).Return(backend.Ban{}, false, nil)

				c.Convey("when fetching the TMI room succeeds", func() {
					tmiRoom := backend.RoomProperties{
						GlobalBannedWordsOptout: true,
					}
					backenderMock.On("GetRoomProperties", mock.Anything, channelID).Return(tmiRoom, true, nil)

					c.Convey("when fetching global banned words succeeds", func() {
						globalBannedWords := []string{"outdated"}
						backenderMock.On("GetGlobalBannedWords", mock.Anything, true).Return(globalBannedWords, nil)

						c.Convey("when fetching channel banned words succeeds", func() {
							channelBannedWords := [][]string{{"long"}}
							backenderMock.On("GetChannelBannedWords", mock.Anything, channelID).Return(channelBannedWords, nil)

							censoredBody := "Jebaited KAPPA *** Jebaited POGCHAMP OVERRATED Jebaited *** HAVE WE WAITED Jebaited NOW WE JEBAITED Jebaited"

							c.Convey("when fetching emote sets succeeds", func() {
								emoteSets := []int{420}
								backenderMock.On("GetUserEmoteSets", mock.Anything, userID).Return(emoteSets, nil)

								c.Convey("when parsing emoticons succeeds", func() {
									emoticons := []models.MessageContentEmoticon{
										{
											Start: 0,
											End:   6,
											ID:    "69",
										},
									}
									backenderMock.On("ParseEmoticons", mock.Anything, censoredBody, emoteSets).Return(emoticons, nil)

									c.Convey("when fetching badges succeeds", func() {
										badges := []models.UserBadge{
											{
												ID:      "LUL",
												Version: "0",
											},
										}
										backenderMock.On("GetUserBadges", mock.Anything, userID, &channelID).Return(badges, nil)

										c.Convey("when fetching the TMI user succeeds", func() {
											tmiUser := backend.TMIUser{
												Color: "blue",
											}
											backenderMock.On("GetTMIUser", mock.Anything, userID).Return(tmiUser, true, nil)

											c.Convey("when fetching the site user succeeds", func() {
												siteUser := backend.SiteUser{
													Login:       "wimpy",
													DisplayName: "(∩｀-´)⊃━☆ﾟ.*･｡ﾟ",
												}
												backenderMock.On("GetSiteUser", mock.Anything, userID).Return(siteUser, true, nil)

												c.Convey("when the message passes AutoMod", func() {
													backenderMock.On("AutoModCheckMessage", mock.Anything, userID, channelID, censoredBody).
														Return(backend.AutoModResponse{IsClean: true}, nil)

													testutils.RunTest(t, s, testutils.APITest{
														URL:    "http://localhost:80/v1/messages/parse",
														Method: "POST",
														BodyJSON: map[string]interface{}{
															"channel_id": channelID,
															"user_id":    userID,
															"body":       messageBody,
														},
														ExpectedStatus: http.StatusOK,
														Expected: map[string]interface{}{
															"status": "ok",
															"message": map[string]interface{}{
																"body": censoredBody,
																"user_badges": []interface{}{
																	map[string]interface{}{
																		"id":      "LUL",
																		"version": "0",
																	},
																},
																"emoticons": []interface{}{
																	map[string]interface{}{
																		"id":    "69",
																		"begin": float64(0),
																		"end":   float64(6),
																	},
																},
																"user_color":        tmiUser.Color,
																"user_login":        siteUser.Login,
																"user_display_name": siteUser.DisplayName,
															},
														},
													})

													backenderMock.AssertExpectations(t)
												})
											})
										})
									})
								})
							})
						})
					})
				})
			})
		})
	})
}
