package app

import (
	"errors"
	"net/http"
	"strconv"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"

	"golang.org/x/net/context"
)

var (
	errInvalidRequestingUser   = errors.New("requesting_user_id is not a valid user")
	errForbiddenRequestingUser = errors.New("requesting user is not authorized to perform this action")
)

func (h *handlers) GetMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.GetModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateGetModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetMod(ctx, params.ChannelID, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, &api.GetModResponse{
		IsMod: exists,
	})
}

func validateGetModParams(params api.GetModRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}
	return nil
}

func (h *handlers) ListMods(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListModsRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListModsParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	mods, err := h.Backend.ListMods(ctx, params.ChannelID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}
	resp := &api.ListModsResponse{Mods: []string{}}
	for _, mod := range mods {
		resp.Mods = append(resp.Mods, mod.UserID)
	}
	gojiplus.ServeJSON(rw, req, resp)
}

func validateListModsParams(params api.ListModsRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	return nil
}

func (h *handlers) AddMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// business logic forbidding adding mods
	siteUser, exists, err := h.Backend.GetSiteUser(ctx, params.RequestingUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusBadRequest)
		return
	}

	if params.RequestingUserID != params.ChannelID &&
		!siteUser.IsStaff && !siteUser.IsAdmin {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// Err if target is banned in channel
	_, exists, err = h.Backend.GetBan(ctx, params.ChannelID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeTargetUserBanned, http.StatusBadRequest)
		return
	}

	// Err if target is already a mod
	_, exists, err = h.Backend.GetMod(ctx, params.ChannelID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if exists {
		serveError(rw, req, api.ErrCodeTargetUserAlreadyMod, http.StatusBadRequest)
		return
	}

	err = h.Backend.AddMod(ctx, params.ChannelID, params.TargetUserID)
	if err == backend.ErrTargetUserAlreadyMod {
		serveError(rw, req, api.ErrCodeTargetUserAlreadyMod, http.StatusBadRequest)
		return
	} else if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}
	rw.WriteHeader(http.StatusOK)
}

func validateAddModParams(params api.AddModRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	if requestingUserID, err := strconv.ParseInt(params.RequestingUserID, 10, 64); err != nil {
		return errors.New("requesting_user_id must be integer")
	} else if requestingUserID <= 0 {
		return errors.New("requesting_user_id must be positive")
	}
	return nil
}

func (h *handlers) RemoveMod(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveModRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveModParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// business logic forbidding removing mods
	siteUser, exists, err := h.Backend.GetSiteUser(ctx, params.RequestingUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusBadRequest)
		return
	}

	if params.RequestingUserID != params.TargetUserID && params.RequestingUserID != params.ChannelID &&
		!siteUser.IsStaff && !siteUser.IsAdmin {
		serveError(rw, req, api.ErrCodeRequestingUserNotPermitted, http.StatusForbidden)
		return
	}

	// Err if target is not already a mod
	_, exists, err = h.Backend.GetMod(ctx, params.ChannelID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeTargetUserNotMod, http.StatusBadRequest)
		return
	}

	err = h.Backend.RemoveMod(ctx, params.ChannelID, params.TargetUserID)
	if err == backend.ErrTargetUserNotMod {
		serveError(rw, req, api.ErrCodeTargetUserNotMod, http.StatusBadRequest)
		return
	} else if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}
	rw.WriteHeader(http.StatusOK)
}

func validateRemoveModParams(params api.RemoveModRequest) error {
	if channelID, err := strconv.ParseInt(params.ChannelID, 10, 64); err != nil {
		return errors.New("channel_id must be integer")
	} else if channelID <= 0 {
		return errors.New("channel_id must be positive")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}
	if requestingUserID, err := strconv.ParseInt(params.RequestingUserID, 10, 64); err != nil {
		return errors.New("requesting_user_id must be integer")
	} else if requestingUserID <= 0 {
		return errors.New("requesting_user_id must be positive")
	}
	return nil
}
