package app

import (
	"net/http"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestHandlersGetMod(t *testing.T) {
	c.Convey("GetMod", t, func() {
		stats, _ := statsd.NewNoopClient()
		backenderMock := &backend.MockBackender{}
		userID := "9999"
		channelID := "10001"
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		c.Convey("with invalid request", func() {
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/get",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": "invalid garbage",
					"user_id":    "asdfasdf",
				},
				ExpectedStatus: http.StatusBadRequest,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("without mod relationship", func() {
			backenderMock.On("GetMod", mock.Anything, channelID, userID).Return(backend.Mod{}, false, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/get",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": channelID,
					"user_id":    userID,
				},
				ExpectedStatus: http.StatusOK,
				Expected: map[string]interface{}{
					"is_mod": false,
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with mod relationship", func() {
			backenderMock.On("GetMod", mock.Anything, channelID, userID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    userID,
			}, true, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/get",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": channelID,
					"user_id":    userID,
				},
				ExpectedStatus: http.StatusOK,
				Expected: map[string]interface{}{
					"is_mod": true,
				},
			})
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestHandlersListMods(t *testing.T) {
	c.Convey("ListMods", t, func() {
		stats, _ := statsd.NewNoopClient()
		backenderMock := &backend.MockBackender{}
		channelID := "10001"
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		c.Convey("with invalid request", func() {
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/list",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": "invalid garbage",
				},
				ExpectedStatus: http.StatusBadRequest,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("without empty response", func() {
			backenderMock.On("ListMods", mock.Anything, channelID).Return([]backend.Mod{}, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/list",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": channelID,
				},
				ExpectedStatus: http.StatusOK,
				Expected: map[string]interface{}{
					"mods": []interface{}{},
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("with mod relationship", func() {
			backenderMock.On("ListMods", mock.Anything, channelID).Return([]backend.Mod{
				backend.Mod{UserID: "111"},
				backend.Mod{UserID: "222"},
			}, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/list",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": channelID,
				},
				ExpectedStatus: http.StatusOK,
				Expected: map[string]interface{}{
					"mods": []interface{}{"111", "222"},
				},
			})
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestHandlersAddMod(t *testing.T) {
	c.Convey("AddMod", t, func() {
		stats, _ := statsd.NewNoopClient()
		backenderMock := &backend.MockBackender{}
		requestingUserID := "5555"
		targetUserID := "9999"
		channelID := "10001"
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		c.Convey("with invalid request", func() {
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": "invalid garbage",
					"user_id":    "asdfasdf",
				},
				ExpectedStatus: http.StatusBadRequest,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is the broadcaster", func() {
			requestingUserID = channelID
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, false, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{}, false, nil).Once()
			backenderMock.On("AddMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is staff", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, false, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{}, false, nil).Once()
			backenderMock.On("AddMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is admin", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsAdmin: true,
			}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, false, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{}, false, nil).Once()
			backenderMock.On("AddMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is not authorized", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, true, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusForbidden,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeRequestingUserNotPermitted,
					"status": float64(http.StatusForbidden),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requesting_user_id is invalid", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, false, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeRequestingUserNotFound,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the target user is already a mod", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, false, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeTargetUserAlreadyMod,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the target user is already a mod, but it doesn't show up until actually calling Backend.AddMod", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, false, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{}, false, nil).Once()
			backenderMock.On("AddMod", mock.Anything, channelID, targetUserID).Return(backend.ErrTargetUserAlreadyMod).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeTargetUserAlreadyMod,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the target user is banned", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetBan", mock.Anything, channelID, targetUserID).Return(backend.Ban{}, true, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/add",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeTargetUserBanned,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})
	})
}

func TestHandlersRemoveMod(t *testing.T) {
	c.Convey("RemoveMod", t, func() {
		stats, _ := statsd.NewNoopClient()
		backenderMock := &backend.MockBackender{}
		requestingUserID := "5555"
		targetUserID := "9999"
		channelID := "10001"
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		c.Convey("with invalid request", func() {
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id": "invalid garbage",
					"user_id":    "asdfasdf",
				},
				ExpectedStatus: http.StatusBadRequest,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is the broadcaster", func() {
			requestingUserID = channelID
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			backenderMock.On("RemoveMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is staff", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			backenderMock.On("RemoveMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is admin", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsAdmin: true,
			}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			backenderMock.On("RemoveMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is the target", func() {
			targetUserID = requestingUserID
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			backenderMock.On("RemoveMod", mock.Anything, channelID, targetUserID).Return(nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusOK,
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requester is not authorized", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, true, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusForbidden,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeRequestingUserNotPermitted,
					"status": float64(http.StatusForbidden),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the requesting_user_id is invalid", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{}, false, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeRequestingUserNotFound,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the target user is not already a mod", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{}, false, nil).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeTargetUserNotMod,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if the target user is not already a mod, but it doesn't show up until calling Backend.RemoveMod", func() {
			backenderMock.On("GetSiteUser", mock.Anything, requestingUserID).Return(backend.SiteUser{
				IsStaff: true,
			}, true, nil).Once()
			backenderMock.On("GetMod", mock.Anything, channelID, targetUserID).Return(backend.Mod{
				ChannelID: channelID,
				UserID:    targetUserID,
			}, true, nil).Once()
			backenderMock.On("RemoveMod", mock.Anything, channelID, targetUserID).Return(backend.ErrTargetUserNotMod).Once()
			testutils.RunTest(t, s, testutils.APITest{
				URL:    "http://localhost:80/v1/mods/remove",
				Method: "POST",
				BodyJSON: map[string]interface{}{
					"channel_id":         channelID,
					"target_user_id":     targetUserID,
					"requesting_user_id": requestingUserID,
				},
				ExpectedStatus: http.StatusBadRequest,
				Expected: map[string]interface{}{
					"error":  api.ErrCodeTargetUserNotMod,
					"status": float64(http.StatusBadRequest),
				},
			})
			backenderMock.AssertExpectations(t)
		})
	})
}
