package app

import (
	"errors"
	"fmt"
	"math"
	"net/http"
	"strings"

	glob "github.com/ryanuber/go-glob"

	"golang.org/x/net/context"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
)

const (
	defaultReservedPerPage = 25
	maxReservedPerPage     = 100
)

func (h *handlers) isReservedName(ctx context.Context, name string) (bool, error) {
	lowerName := strings.ToLower(name)
	reservedNames, err := h.Backend.ListCommunityReservedNames(ctx)
	if err != nil {
		return false, err
	}

	for _, pattern := range reservedNames {
		if matches := glob.Glob(pattern, lowerName); matches {
			return true, nil
		}
	}

	return false, nil
}

func (h *handlers) ListCommunityReservedNames(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListCommunityReservedNamesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListCommunityReservedNamesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	if params.PerPage == 0 {
		params.PerPage = defaultReservedPerPage
	}

	// Zero-index-afy
	params.Page = params.Page - 1

	allReservedNames, err := h.Backend.ListCommunityReservedNames(ctx)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	resp := reservedSliceToResponse(allReservedNames, params.Page, params.PerPage, params.Filter)

	gojiplus.ServeJSON(rw, req, &resp)
}

func validateListCommunityReservedNamesParams(params api.ListCommunityReservedNamesRequest) error {
	if params.Page < 1 {
		return errors.New("page must not be greater than 1")
	}
	if params.PerPage < 0 || params.PerPage > maxReservedPerPage {
		return errors.New(fmt.Sprintf("per_page must be between 0 and %d", maxReservedPerPage))
	}
	return nil
}

func (h *handlers) DeleteCommunityReservedName(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.DeleteCommunityReservedNameRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateDeleteCommunityReservedNameParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	err := h.Backend.DeleteCommunityReservedName(ctx, params.ReservedName)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateDeleteCommunityReservedNameParams(params api.DeleteCommunityReservedNameRequest) error {
	if params.ReservedName == "" {
		return errors.New("reserved_name must not be empty")
	}
	return nil
}

func (h *handlers) CreateCommunityReservedName(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.CreateCommunityReservedNameRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateCreateCommunityReservedNameParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	lowercaseName := strings.ToLower(params.ReservedName)
	err := h.Backend.CreateCommunityReservedName(ctx, lowercaseName)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateCreateCommunityReservedNameParams(params api.CreateCommunityReservedNameRequest) error {
	if params.ReservedName == "" {
		return errors.New("reserved_name must not be empty")
	}
	return nil
}

func reservedSliceToResponse(raw []string, page, perPage int, filter string) api.ListCommunityReservedNamesResponse {
	out := api.ListCommunityReservedNamesResponse{
		ReservedNames: []api.CommunityReservedName{},
	}

	filtered := []string{}
	if filter == "" {
		filtered = raw
	} else {
		for _, name := range raw {
			if strings.Contains(name, filter) {
				filtered = append(filtered, name)
			}
		}
	}

	filteredCount := len(filtered)
	finalPageCount := float64(filteredCount) / float64(perPage)
	out.TotalPages = int(math.Ceil(finalPageCount))

	startPos := perPage * page
	endPos := int(math.Min(float64(startPos+perPage), float64(filteredCount)))
	for _, name := range filtered[startPos:endPos] {
		out.ReservedNames = append(out.ReservedNames, api.CommunityReservedName{
			Name: name,
		})
	}

	return out
}
