package app

import (
	"context"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/backend"
)

func TestListCommunityReservedNames(t *testing.T) {
	c.Convey("test ListCommunityReservedNames handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/reserved_names/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"page": 1,
			},
			ExpectedStatus: http.StatusOK,
		}
		c.Convey("with no filter", func() {
			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
					"two",
					"three",
				}, nil).Once()
			apitest.Expected = map[string]interface{}{
				"total_pages": float64(1),
				"reserved_names": []interface{}{
					map[string]interface{}{
						"reserved_name": "one",
					},
					map[string]interface{}{
						"reserved_name": "two",
					},
					map[string]interface{}{
						"reserved_name": "three",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with a filter", func() {
			apitest.BodyJSON["filter"] = "o"

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
					"two",
					"three",
				}, nil).Once()
			apitest.Expected = map[string]interface{}{
				"total_pages": float64(1),
				"reserved_names": []interface{}{
					map[string]interface{}{
						"reserved_name": "one",
					},
					map[string]interface{}{
						"reserved_name": "two",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with pages, page 1", func() {
			apitest.BodyJSON["per_page"] = 2

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
					"two",
					"three",
					"four",
				}, nil).Once()
			apitest.Expected = map[string]interface{}{
				"total_pages": float64(2),
				"reserved_names": []interface{}{
					map[string]interface{}{
						"reserved_name": "one",
					},
					map[string]interface{}{
						"reserved_name": "two",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with pages, page 2", func() {
			apitest.BodyJSON["per_page"] = 2
			apitest.BodyJSON["page"] = 2

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
					"two",
					"three",
					"four",
				}, nil).Once()
			apitest.Expected = map[string]interface{}{
				"total_pages": float64(2),
				"reserved_names": []interface{}{
					map[string]interface{}{
						"reserved_name": "three",
					},
					map[string]interface{}{
						"reserved_name": "four",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with pages and filter, page 2", func() {
			apitest.BodyJSON["per_page"] = 2
			apitest.BodyJSON["page"] = 2
			apitest.BodyJSON["filter"] = "match"

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"nope 1",
					"nope 2",
					"match 1",
					"match 2",
					"nope 3",
					"match 3",
				}, nil).Once()
			apitest.Expected = map[string]interface{}{
				"total_pages": float64(2),
				"reserved_names": []interface{}{
					map[string]interface{}{
						"reserved_name": "match 3",
					},
				},
			}
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with a bad page number", func() {
			apitest.BodyJSON["page"] = -1

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
				}, nil).Once()
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})
		c.Convey("with a bad per page", func() {
			apitest.BodyJSON["per_page"] = -1

			backenderMock.On("ListCommunityReservedNames", mock.Anything).
				Return([]string{
					"one",
				}, nil).Once()
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

	})
}

func TestCreateCommunityReservedName(t *testing.T) {
	c.Convey("test CreateCommunityReservedName handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})
		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/reserved_names/create",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"reserved_name": "123",
			},
			ExpectedStatus: http.StatusOK,
		}
		c.Convey("with valid parameters", func() {
			backenderMock.On("CreateCommunityReservedName", mock.Anything, "123").
				Return(nil).Once()
			testutils.RunTest(t, s, apitest)
		})
		c.Convey("with no parameters", func() {
			delete(apitest.BodyJSON, "reserved_name")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})
	})
}

func TestDeleteCommunityReservedName(t *testing.T) {
	c.Convey("test DeleteCommunityReservedName handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})
		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/reserved_names/delete",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"reserved_name": "123",
			},
			ExpectedStatus: http.StatusOK,
		}
		c.Convey("with valid parameters", func() {
			backenderMock.On("DeleteCommunityReservedName", mock.Anything, "123").
				Return(nil).Once()
			testutils.RunTest(t, s, apitest)
		})
		c.Convey("with no parameters", func() {
			delete(apitest.BodyJSON, "reserved_name")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})
	})
}

func TestIsReservedCommunityName(t *testing.T) {
	c.Convey("test isReservedName", t, func() {
		backenderMock := &backend.MockBackender{}
		h := &handlers{
			Backend: backenderMock,
		}
		ctx := context.Background()
		backenderMock.On("ListCommunityReservedNames", ctx).
			Return([]string{
				"fullmatch",
				"prefixmatch*",
				"*suffixmatch",
				"*surroundmatch*",
			}, nil)

		c.Convey("full matches are reserved", func() {
			reserved, _ := h.isReservedName(ctx, "fullmatch")
			c.So(reserved, c.ShouldEqual, true)
		})
		c.Convey("non-full matches are not reserved", func() {
			reserved, _ := h.isReservedName(ctx, "notafullmatchatall")
			c.So(reserved, c.ShouldEqual, false)
		})
		c.Convey("prefix matches are reserved", func() {
			reserved, _ := h.isReservedName(ctx, "prefixmatcheh")
			c.So(reserved, c.ShouldEqual, true)
			reserved, _ = h.isReservedName(ctx, "prefixmatch")
			c.So(reserved, c.ShouldEqual, true)
		})
		c.Convey("prefix matches are not reserved if there's extra stuff", func() {
			reserved, _ := h.isReservedName(ctx, "notprefixmatch")
			c.So(reserved, c.ShouldEqual, false)
		})
		c.Convey("suffix matches are reserved", func() {
			reserved, _ := h.isReservedName(ctx, "yosuffixmatch")
			c.So(reserved, c.ShouldEqual, true)
			reserved, _ = h.isReservedName(ctx, "suffixmatch")
			c.So(reserved, c.ShouldEqual, true)
		})
		c.Convey("suffix matches are not reserved if there's extra stuff", func() {
			reserved, _ := h.isReservedName(ctx, "suffixmatchjk")
			c.So(reserved, c.ShouldEqual, false)
		})
		c.Convey("surround matches are reserved", func() {
			reserved, _ := h.isReservedName(ctx, "heresasurroundmatch")
			c.So(reserved, c.ShouldEqual, true)
			reserved, _ = h.isReservedName(ctx, "heresasurroundmatchm8")
			c.So(reserved, c.ShouldEqual, true)
			reserved, _ = h.isReservedName(ctx, "surroundmatch")
			c.So(reserved, c.ShouldEqual, true)
		})
		c.Convey("surround matches are not reserved if they're spelled wrong", func() {
			reserved, _ := h.isReservedName(ctx, "butitsasurroundsoundmatch")
			c.So(reserved, c.ShouldEqual, false)
		})
	})
}
