package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"

	"golang.org/x/net/context"
)

const defaultTopLimit = 20

func (h *handlers) TopCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	var err error
	params := api.TopCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateTopCommunitiesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// determine offset and limit
	var offset int
	if params.Cursor != "" {
		offset, err = api.CursorToInt(params.Cursor)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
			return
		}
	}

	limit := params.Limit
	if limit == 0 {
		limit = defaultTopLimit
	}

	// fetch full list of live communities from Jax
	liveCommunities, err := h.Backend.LiveCommunities(ctx)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	// if only looking for featured communities, filter out non-featured ones
	// before doing pagination
	if params.FeaturedOnly {
		liveCommunities, err = h.filterLiveCommunitiesToFeaturedCommunities(ctx, liveCommunities)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	if params.Language != "" {
		liveCommunities, err = h.filterLiveCommunitiesByLanguage(ctx, liveCommunities, params.Language)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
			return
		}
	}

	count := len(liveCommunities)

	// take just the offset -> limit
	part := []backend.LiveCommunityStats{}
	if offset < len(liveCommunities) {
		end := min(len(liveCommunities), offset+limit)
		part = liveCommunities[offset:end]
	}

	topCommunities, err := h.topCommunityResultsFromLiveCommunityStats(ctx, part)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	nextCursor := ""
	nextOffset := len(part) + offset
	if nextOffset < count {
		nextCursor = api.IntToCursorString(nextOffset)
	}

	gojiplus.ServeJSON(rw, req, api.TopCommunitiesResponse{
		Results: topCommunities,
		Total:   count,
		Cursor:  nextCursor,
	})
}

// topCommunityResultsFromLiveCommunityStats takes a slice of LiveCommunityStats
// (community ID, # viewers, # channels) and populates it with the displaynames
// and avatar image URLs for each community
func (h *handlers) topCommunityResultsFromLiveCommunityStats(ctx context.Context, part []backend.LiveCommunityStats) ([]api.TopCommunityResult, error) {
	// get slice of just community IDs
	communityIDs := []string{}
	for _, result := range part {
		if result.CommunityID != "" {
			communityIDs = append(communityIDs, result.CommunityID)
		}
	}

	// lookup details for each community
	// returns map from community ID to details
	communities, err := h.Backend.BulkGetCommunities(ctx, communityIDs)
	if err != nil {
		return nil, err
	}

	// generate results
	topCommunities := []api.TopCommunityResult{}
	for _, result := range part {
		if community, ok := communities[result.CommunityID]; ok {
			topCommunities = append(topCommunities, api.TopCommunityResult{
				ID:             community.CommunityID,
				Name:           community.Name,
				DisplayName:    community.DisplayName,
				AvatarImageURL: h.communityImageURL(community.CommunityID, community.AvatarImageName, api.ImageTypeAvatar),
				Viewers:        result.Viewers,
				Channels:       result.Channels,
			})
		}
	}
	return topCommunities, nil
}

func validateTopCommunitiesParams(params api.TopCommunitiesRequest) error {
	if params.Limit < 0 || params.Limit > maxLimit {
		return errors.New("invalid limit")
	}
	return nil
}

func min(x, y int) int {
	if x < y {
		return x
	}
	return y
}

// filterLiveCommunitiesByLanguage takes the full list of live
// communities, and filters it by language
func (h *handlers) filterLiveCommunitiesByLanguage(ctx context.Context, liveCommunities []backend.LiveCommunityStats, language string) ([]backend.LiveCommunityStats, error) {
	// get all featured communities
	communityIDs, err := h.Backend.GetCommunityIDsByLanguage(ctx, language)
	if err != nil {
		return nil, err
	}

	// convert to map for easy access
	communityIDsMap := map[string]struct{}{}
	for _, id := range communityIDs {
		communityIDsMap[id] = struct{}{}
	}

	resp := []backend.LiveCommunityStats{}

	// add live communities with the right language to the response
	for _, lc := range liveCommunities {
		if _, ok := communityIDsMap[lc.CommunityID]; ok {
			resp = append(resp, lc)
		}
	}

	return resp, nil
}
