package app

import (
	"errors"
	"net/http"
	"testing"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestTopCommunities(t *testing.T) {
	c.Convey("test TopCommunities handler", t, func() {

		var (
			testCommunityA = backend.Community{
				CommunityID:      "abc",
				Name:             "mycommunity",
				DisplayName:      "mycommunity",
				OwnerUserID:      "111",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "this@is.fake",
				BannerImageName:  "asdfasdf.png",
				AvatarImageName:  "123.png",
			}
			testCommunityB = backend.Community{
				CommunityID:      "querty",
				Name:             "blahblah",
				DisplayName:      "Blah Blah",
				OwnerUserID:      "222",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "rules",
				Language:         "EN",
				Email:            "none",
				BannerImageName:  "testest.png",
				AvatarImageName:  "xkcd.png",
			}
			testCommunityC = backend.Community{
				CommunityID:      "lol",
				Name:             "hellothere",
				DisplayName:      "hellothere",
				OwnerUserID:      "444",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "a@b.c",
				BannerImageName:  "lolol.png",
				AvatarImageName:  "abc.png",
			}
		)

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
			Conf: config.Config{
				Environment: "production",
			},
		})

		liveCommunitiesResp := []backend.LiveCommunityStats{
			backend.LiveCommunityStats{
				CommunityID: testCommunityA.CommunityID,
				Viewers:     25,
				Channels:    22,
			},
			backend.LiveCommunityStats{
				CommunityID: testCommunityB.CommunityID,
				Viewers:     21,
				Channels:    12,
			},
			backend.LiveCommunityStats{
				CommunityID: testCommunityB.CommunityID,
				Viewers:     15,
				Channels:    3,
			},
		}
		bulkCommunitiesResp := map[string]backend.Community{
			testCommunityA.CommunityID: testCommunityA,
			testCommunityB.CommunityID: testCommunityB,
			testCommunityC.CommunityID: testCommunityC,
		}

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/communities/top",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"limit":         10,
				"featured_only": false,
				"language":      "",
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with successful response for featured community IDs", func() {
			backenderMock.On("LiveCommunities", mock.Anything).
				Return(liveCommunitiesResp, nil).Once()

			c.Convey("with successful response for bulk get", func() {
				backenderMock.On("BulkGetCommunities", mock.Anything, mock.Anything).Return(bulkCommunitiesResp, nil).Once()

				apitest.Expected = map[string]interface{}{
					"cursor": "",
					"results": []interface{}{
						map[string]interface{}{
							"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/abc/123-185x258.png",
							"channels":         float64(22),
							"id":               "abc",
							"name":             "mycommunity",
							"display_name":     "mycommunity",
							"viewers":          float64(25),
						},
						map[string]interface{}{
							"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/querty/xkcd-185x258.png",
							"channels":         float64(12),
							"id":               "querty",
							"name":             "blahblah",
							"display_name":     "Blah Blah",
							"viewers":          float64(21),
						},
						map[string]interface{}{
							"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/querty/xkcd-185x258.png",
							"channels":         float64(3),
							"id":               "querty",
							"name":             "blahblah",
							"display_name":     "Blah Blah",
							"viewers":          float64(15),
						},
					},
					"total": float64(3),
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("with error response for bulk get", func() {
				backenderMock.On("BulkGetCommunities", mock.Anything, mock.Anything).
					Return(nil, errors.New("error")).Once()

				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})
		})

		c.Convey("with error response for featured community IDs", func() {
			backenderMock.On("LiveCommunities", mock.Anything).
				Return(nil, errors.New("error")).Once()

			apitest.ExpectedStatus = http.StatusInternalServerError
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})
	})
}
