package app

import (
	"errors"
	"net/http"
	"strconv"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/zuma/app/api"

	"golang.org/x/net/context"
)

func (h *handlers) ListUserBlocks(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListUserBlocksParams{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListUserBlocksParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	blocks, err := h.Backend.ListUserBlocks(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	gojiplus.ServeJSON(rw, req, api.ListUserBlocksResponse{
		BlockedUserIDs: blocks,
	})
}

func validateListUserBlocksParams(params api.ListUserBlocksParams) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}

	return nil
}

func (h *handlers) AddUserBlock(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddUserBlockParams{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddUserBlockParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	msg, success, err := h.Backend.AddUserBlock(ctx, params.UserID, params.TargetUserID, params.Reason, params.SourceContext)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !success {
		gojiplus.ServeJSONWithStatus(rw, req, api.AddUserBlockResponse{
			Status:  http.StatusUnprocessableEntity,
			Error:   "Unprocessable Entity",
			Message: msg,
		}, http.StatusUnprocessableEntity)
		return
	}

	if err := h.Backend.PublishAddUserBlock(ctx, params.UserID, params.TargetUserID, params.Reason, params.SourceContext); err != nil {
		logx.Error(ctx, err)
	}

	gojiplus.ServeJSON(rw, req, api.AddUserBlockResponse{
		Status: http.StatusOK,
	})
}

func validateAddUserBlockParams(params api.AddUserBlockParams) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}

	if params.TargetUserID == "" {
		return errors.New("target_user_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}

	return nil
}

func (h *handlers) RemoveUserBlock(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveUserBlockParams{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveUserBlockParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	err := h.Backend.RemoveUserBlock(ctx, params.UserID, params.TargetUserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	if err := h.Backend.PublishRemoveUserBlock(ctx, params.UserID, params.TargetUserID); err != nil {
		logx.Error(ctx, err)
	}

	rw.WriteHeader(http.StatusOK)
}

func validateRemoveUserBlockParams(params api.RemoveUserBlockParams) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if userID, err := strconv.ParseInt(params.UserID, 10, 64); err != nil {
		return errors.New("user_id must be integer")
	} else if userID <= 0 {
		return errors.New("user_id must be positive")
	}

	if params.TargetUserID == "" {
		return errors.New("target_user_id must not be empty")
	}
	if targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64); err != nil {
		return errors.New("target_user_id must be integer")
	} else if targetUserID <= 0 {
		return errors.New("target_user_id must be positive")
	}

	return nil
}
