package app

import (
	"errors"
	"net/http"
	"testing"

	"github.com/cactus/go-statsd-client/statsd"
	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestListUserBlocks(t *testing.T) {
	c.Convey("test ListUserBlocks handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/blocks/get",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id": "123",
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with empty user_id", func() {
			delete(apitest.BodyJSON, "user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with valid args", func() {

			c.Convey("with successful response", func() {
				backenderMock.On("ListUserBlocks", mock.Anything, "123").
					Return([]string{
						"111",
						"222",
						"333",
					}, nil).Once()
				apitest.Expected = map[string]interface{}{
					"blocked_user_ids": []interface{}{
						"111",
						"222",
						"333",
					},
				}
				testutils.RunTest(t, s, apitest)
			})
		})
	})
}

func TestAddUserBlock(t *testing.T) {
	var (
		userID        = "123"
		targetUserID  = "456"
		reason        = "spam"
		sourceContext = "whisper"
	)
	stats, _ := statsd.NewNoopClient()

	c.Convey("test AddUserBlock handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/blocks/add",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id":        userID,
				"target_user_id": targetUserID,
				"reason":         reason,
				"source_context": sourceContext,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with empty user_id", func() {
			delete(apitest.BodyJSON, "user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with empty target_user_id", func() {
			delete(apitest.BodyJSON, "target_user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with valid args", func() {
			backenderMock.On("PublishAddUserBlock", mock.Anything, userID, targetUserID, reason, sourceContext).
				Return(nil).Once()

			c.Convey("with error response", func() {
				backenderMock.On("AddUserBlock", mock.Anything, userID, targetUserID, reason, sourceContext).
					Return("", false, errors.New("foo")).Once()
				apitest.ExpectedStatus = http.StatusInternalServerError
				testutils.RunTest(t, s, apitest)
			})

			c.Convey("with unsuccessful response", func() {
				backenderMock.On("AddUserBlock", mock.Anything, userID, targetUserID, reason, sourceContext).
					Return("internal error message here", false, nil).Once()
				apitest.ExpectedStatus = http.StatusUnprocessableEntity
				apitest.Expected = map[string]interface{}{
					"status":  float64(http.StatusUnprocessableEntity),
					"error":   "Unprocessable Entity",
					"message": "internal error message here",
				}
				testutils.RunTest(t, s, apitest)
			})

			c.Convey("with successful response", func() {
				backenderMock.On("AddUserBlock", mock.Anything, userID, targetUserID, reason, sourceContext).
					Return("", true, nil).Once()
				testutils.RunTest(t, s, apitest)
			})
		})
	})
}

func TestRemoveUserBlock(t *testing.T) {
	var (
		userID       = "123"
		targetUserID = "456"
	)
	stats, _ := statsd.NewNoopClient()

	c.Convey("test RemoveUserBlock handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Conf:    config.Config{},
			Stats:   stats,
			Backend: backenderMock,
		})

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/blocks/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id":        userID,
				"target_user_id": targetUserID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with empty user_id", func() {
			delete(apitest.BodyJSON, "user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with empty target_user_id", func() {
			delete(apitest.BodyJSON, "target_user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("with valid args", func() {
			backenderMock.On("PublishRemoveUserBlock", mock.Anything, userID).
				Return(nil).Once()

			c.Convey("with successful response", func() {
				backenderMock.On("RemoveUserBlock", mock.Anything, userID, targetUserID).
					Return("", true, nil).Once()
				testutils.RunTest(t, s, apitest)
			})
		})
	})
}
