package app

import (
	"errors"
	"net/http"

	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/backend"

	"golang.org/x/net/context"
)

func (h *handlers) ListUserFollowedCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.ListUserFollowedCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateListUserFollowedCommunitiesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// specify limit
	limit := defaultFollowersLimit
	if params.Limit > 0 && params.Limit <= maxFollowersLimit {
		limit = params.Limit
	} else if params.Limit > maxFollowersLimit {
		limit = maxFollowersLimit
	}

	_, exists, err := h.Backend.GetSiteUser(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusNotFound)
		return
	}

	followedCommunities, newCursor, err := h.Backend.ListUserFollowedCommunities(ctx, params.UserID, params.Cursor, limit)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	total, err := h.Backend.CountUserFollowedCommunities(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	communityIDs := []string{}
	for _, fc := range followedCommunities {
		communityIDs = append(communityIDs, fc.CommunityID)
	}

	communitiesMap, err := h.Backend.BulkGetCommunities(ctx, communityIDs)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	communities := []api.FollowedCommunity{}
	for _, fc := range followedCommunities {
		if community, ok := communitiesMap[fc.CommunityID]; ok && !community.TOSBanned {
			c := h.apiCommunityFromBackendCommunity(community)
			communities = append(communities, api.FollowedCommunity{
				CommunityID:    c.CommunityID,
				Name:           c.Name,
				DisplayName:    c.DisplayName,
				AvatarImageURL: c.AvatarImageURL,
				CreatedAt:      fc.CreatedAt.UTC(),
			})
		}
	}

	gojiplus.ServeJSON(rw, req, &api.ListUserFollowedCommunitiesResponse{
		FollowedCommunities: communities,
		Cursor:              newCursor,
		Total:               total,
	})
}

func validateListUserFollowedCommunitiesParams(params api.ListUserFollowedCommunitiesRequest) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if params.Limit < 0 {
		return errors.New("limit must not be negative")
	}
	return nil
}

func (h *handlers) TopUserFollowedCommunities(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.TopUserFollowedCommunitiesRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateTopUserFollowedCommunitiesParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	// determine offset and limit
	var err error
	var offset int
	if params.Cursor != "" {
		offset, err = api.CursorToInt(params.Cursor)
		if err != nil {
			gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
			return
		}
	}

	limit := params.Limit
	if limit == 0 {
		limit = defaultTopLimit
	}

	_, exists, err := h.Backend.GetSiteUser(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusNotFound)
		return
	}

	// fetch full list of live communities from Jax
	liveCommunities, err := h.Backend.LiveCommunities(ctx)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	followedCommunities, err := h.Backend.AllUserFollowedCommunities(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	followedCommunitiesMap := map[string]struct{}{}
	for _, fc := range followedCommunities {
		followedCommunitiesMap[fc.CommunityID] = struct{}{}
	}

	filteredLiveCommunities := []backend.LiveCommunityStats{}
	for _, lc := range liveCommunities {
		if _, ok := followedCommunitiesMap[lc.CommunityID]; ok {
			filteredLiveCommunities = append(filteredLiveCommunities, lc)
		}
	}

	count := len(filteredLiveCommunities)

	// take just the offset -> limit
	part := []backend.LiveCommunityStats{}
	if offset < len(filteredLiveCommunities) {
		end := min(len(filteredLiveCommunities), offset+limit)
		part = filteredLiveCommunities[offset:end]
	}

	topCommunities, err := h.topCommunityResultsFromLiveCommunityStats(ctx, part)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	nextCursor := ""
	nextOffset := len(part) + offset
	if nextOffset < count {
		nextCursor = api.IntToCursorString(nextOffset)
	}

	gojiplus.ServeJSON(rw, req, api.TopUserFollowedCommunitiesResponse{
		Results: topCommunities,
		Total:   count,
		Cursor:  nextCursor,
	})
}

func validateTopUserFollowedCommunitiesParams(params api.TopUserFollowedCommunitiesRequest) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if params.Limit < 0 || params.Limit > maxLimit {
		return errors.New("invalid limit")
	}
	return nil
}

func (h *handlers) AddUserFollowedCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.AddUserFollowedCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateAddUserFollowedCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetSiteUser(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusNotFound)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	err = h.Backend.AddUserFollowedCommunity(ctx, params.UserID, params.CommunityID)
	if err != nil && err != backend.ErrUserAlreadyFollowsCommunity {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateAddUserFollowedCommunityParams(params api.AddUserFollowedCommunityRequest) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}

func (h *handlers) RemoveUserFollowedCommunity(ctx context.Context, rw http.ResponseWriter, req *http.Request) {
	params := api.RemoveUserFollowedCommunityRequest{}
	if err := gojiplus.ParseJSONFromRequest(req, &params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	} else if err := validateRemoveUserFollowedCommunityParams(params); err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusBadRequest)
		return
	}

	_, exists, err := h.Backend.GetSiteUser(ctx, params.UserID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeRequestingUserNotFound, http.StatusNotFound)
		return
	}

	community, exists, err := h.Backend.GetCommunity(ctx, params.CommunityID)
	if err != nil {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	} else if !exists {
		serveError(rw, req, api.ErrCodeCommunityIDNotFound, http.StatusNotFound)
		return
	}

	if community.TOSBanned {
		serveError(rw, req, api.ErrCodeCommunityTOSBanned, http.StatusNotFound)
		return
	}

	err = h.Backend.RemoveUserFollowedCommunity(ctx, params.UserID, params.CommunityID)
	if err != nil && err != backend.ErrUserNotFollowingCommunity {
		gojiplus.ServeError(ctx, rw, req, err, http.StatusInternalServerError)
		return
	}

	rw.WriteHeader(http.StatusOK)
}

func validateRemoveUserFollowedCommunityParams(params api.RemoveUserFollowedCommunityRequest) error {
	if params.UserID == "" {
		return errors.New("user_id must not be empty")
	}
	if params.CommunityID == "" {
		return errors.New("community_id must not be empty")
	}
	return nil
}
