package app

import (
	"errors"
	"net/http"
	"testing"
	"time"

	c "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/testutils"
	"code.justin.tv/chat/zuma/app/api"
	"code.justin.tv/chat/zuma/app/config"
	"code.justin.tv/chat/zuma/backend"
)

func TestListUserFollowedCommunities(t *testing.T) {
	c.Convey("test ListUserFollowedCommunities handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		userID := "123"

		var (
			testCommunityA = backend.Community{
				CommunityID:      "abc",
				Name:             "mycommunity",
				DisplayName:      "mycommunity",
				OwnerUserID:      "111",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "this@is.fake",
				BannerImageName:  "asdfasdf.png",
				AvatarImageName:  "123.png",
			}
			testCommunityB = backend.Community{
				CommunityID:      "qwerty",
				Name:             "blahblah",
				DisplayName:      "blahblah",
				OwnerUserID:      "222",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "rules",
				Language:         "EN",
				Email:            "none",
				BannerImageName:  "testest.png",
				AvatarImageName:  "xkcd.png",
			}
			testCommunityC = backend.Community{
				CommunityID:      "lol",
				Name:             "hellothere",
				DisplayName:      "hellothere",
				OwnerUserID:      "444",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "a@b.c",
				BannerImageName:  "lolol.png",
				AvatarImageName:  "abc.png",
			}
		)

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/follows/communities/list",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id": userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("with empty user_id", func() {
			delete(apitest.BodyJSON, "user_id")
			apitest.ExpectedStatus = http.StatusBadRequest
			testutils.RunTest(t, s, apitest)
		})

		c.Convey("if user does not exist", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeRequestingUserNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if user exists", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, true, nil).Once()

			c.Convey("with valid args", func() {

				c.Convey("with successful response for list followed communities", func() {
					backenderMock.On("ListUserFollowedCommunities", mock.Anything, userID, mock.Anything, mock.Anything).
						Return([]backend.CommunityFollower{
							backend.CommunityFollower{
								CommunityID: "abc",
								UserID:      "123",
								CreatedAt:   time.Unix(111, 0).UTC(),
							},
							backend.CommunityFollower{
								CommunityID: "qwerty",
								UserID:      "123",
								CreatedAt:   time.Unix(222, 0).UTC(),
							},
							backend.CommunityFollower{
								CommunityID: "lol",
								UserID:      "123",
								CreatedAt:   time.Unix(333, 0).UTC(),
							},
						}, "101010", nil).Once()

					backenderMock.On("CountUserFollowedCommunities", mock.Anything, "123").
						Return(3, nil).Once()

					c.Convey("with successful bulk get response", func() {
						backenderMock.On("BulkGetCommunities", mock.Anything, []string{"abc", "qwerty", "lol"}).
							Return(
								map[string]backend.Community{
									"abc":    testCommunityA,
									"qwerty": testCommunityB,
									"lol":    testCommunityC,
								}, nil).Once()

						apitest.Expected = map[string]interface{}{
							"cursor": "101010",
							"total":  float64(3),
							"followed_communities": []interface{}{
								map[string]interface{}{
									"avatar_image_url": "https://twitch-community-images-staging.s3.amazonaws.com/abc/123.png",
									"community_id":     "abc",
									"name":             "mycommunity",
									"display_name":     "mycommunity",
									"created_at":       "1970-01-01T00:01:51Z",
								},
								map[string]interface{}{
									"avatar_image_url": "https://twitch-community-images-staging.s3.amazonaws.com/qwerty/xkcd.png",
									"community_id":     "qwerty",
									"name":             "blahblah",
									"display_name":     "blahblah",
									"created_at":       "1970-01-01T00:03:42Z",
								},
								map[string]interface{}{
									"avatar_image_url": "https://twitch-community-images-staging.s3.amazonaws.com/lol/abc.png",
									"community_id":     "lol",
									"name":             "hellothere",
									"display_name":     "hellothere",
									"created_at":       "1970-01-01T00:05:33Z",
								},
							},
						}
						testutils.RunTest(t, s, apitest)
					})
				})
			})
		})
	})
}

func TestTopUserFollowedCommunities(t *testing.T) {
	c.Convey("test TopUserFollowedCommunities handler", t, func() {

		var (
			testCommunityA = backend.Community{
				CommunityID:      "abc",
				Name:             "mycommunity",
				DisplayName:      "mycommunity",
				OwnerUserID:      "111",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "this@is.fake",
				BannerImageName:  "asdfasdf.png",
				AvatarImageName:  "123.png",
			}
			testCommunityB = backend.Community{
				CommunityID:      "qwerty",
				Name:             "blahblah",
				DisplayName:      "blahblah",
				OwnerUserID:      "222",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "rules",
				Language:         "EN",
				Email:            "none",
				BannerImageName:  "testest.png",
				AvatarImageName:  "xkcd.png",
			}
			testCommunityC = backend.Community{
				CommunityID:      "lol",
				Name:             "hellothere",
				DisplayName:      "hellothere",
				OwnerUserID:      "444",
				ShortDescription: "short short",
				LongDescription:  "long long long",
				Rules:            "no rules",
				Language:         "EN",
				Email:            "a@b.c",
				BannerImageName:  "lolol.png",
				AvatarImageName:  "abc.png",
			}
		)

		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
			Conf: config.Config{
				Environment: "production",
			},
		})

		liveCommunitiesResp := []backend.LiveCommunityStats{
			backend.LiveCommunityStats{
				CommunityID: testCommunityA.CommunityID,
				Viewers:     25,
				Channels:    22,
			},
			backend.LiveCommunityStats{
				CommunityID: testCommunityB.CommunityID,
				Viewers:     21,
				Channels:    12,
			},
			backend.LiveCommunityStats{
				CommunityID: testCommunityB.CommunityID,
				Viewers:     15,
				Channels:    3,
			},
		}
		bulkCommunitiesResp := map[string]backend.Community{
			testCommunityA.CommunityID: testCommunityA,
			testCommunityB.CommunityID: testCommunityB,
			testCommunityC.CommunityID: testCommunityC,
		}

		userID := "123"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/follows/communities/top",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"limit":   10,
				"user_id": userID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if user does not exist", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeRequestingUserNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if user exists", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, true, nil).Once()

			c.Convey("with successful response for user followed communities and live communities", func() {
				backenderMock.On("LiveCommunities", mock.Anything).
					Return(liveCommunitiesResp, nil).Once()
				backenderMock.On("AllUserFollowedCommunities", mock.Anything, userID).
					Return([]backend.CommunityFollower{
						backend.CommunityFollower{
							CommunityID: "abc",
							UserID:      "123",
							CreatedAt:   time.Unix(111, 0).UTC(),
						},
						backend.CommunityFollower{
							CommunityID: "qwerty",
							UserID:      "123",
							CreatedAt:   time.Unix(222, 0).UTC(),
						},
						backend.CommunityFollower{
							CommunityID: "lol",
							UserID:      "123",
							CreatedAt:   time.Unix(333, 0).UTC(),
						},
					}, nil).Once()

				c.Convey("with successful response for bulk get", func() {
					backenderMock.On("BulkGetCommunities", mock.Anything, mock.Anything).Return(bulkCommunitiesResp, nil).Once()

					apitest.Expected = map[string]interface{}{
						"cursor": "",
						"results": []interface{}{
							map[string]interface{}{
								"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/abc/123-185x258.png",
								"channels":         float64(22),
								"id":               "abc",
								"name":             "mycommunity",
								"display_name":     "mycommunity",
								"viewers":          float64(25),
							},
							map[string]interface{}{
								"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/qwerty/xkcd-185x258.png",
								"channels":         float64(12),
								"id":               "qwerty",
								"name":             "blahblah",
								"display_name":     "blahblah",
								"viewers":          float64(21),
							},
							map[string]interface{}{
								"avatar_image_url": "https://static-cdn.jtvnw.net/community-images/qwerty/xkcd-185x258.png",
								"channels":         float64(3),
								"id":               "qwerty",
								"name":             "blahblah",
								"display_name":     "blahblah",
								"viewers":          float64(15),
							},
						},
						"total": float64(3),
					}
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})
		})
	})
}

func TestAddUserFollowedCommunity(t *testing.T) {
	c.Convey("test AddUserFollowedCommunity handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		userID := "123"
		communityID := "asdfasdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/follows/communities/add",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id":      userID,
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if user does not exist", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeRequestingUserNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if user exists", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, true, nil).Once()

			c.Convey("if community is tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{
						TOSBanned: true,
					}, true, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityTOSBanned,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community does not exist", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityIDNotFound,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community exists and is not tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, true, nil).Once()

				c.Convey("with noop error", func() {
					backenderMock.On("AddUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(backend.ErrUserAlreadyFollowsCommunity).Once()

					apitest.ExpectedStatus = http.StatusOK
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("with no error", func() {
					backenderMock.On("AddUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(nil).Once()

					apitest.ExpectedStatus = http.StatusOK
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("with not noop error", func() {
					backenderMock.On("AddUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusInternalServerError
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})
		})
	})
}

func TestRemoveUserFollowedCommunity(t *testing.T) {
	c.Convey("test RemoveUserFollowedCommunity handler", t, func() {
		backenderMock := &backend.MockBackender{}
		s, _ := New(Params{
			Backend: backenderMock,
		})

		userID := "123"
		communityID := "asdfasdf"

		apitest := testutils.APITest{
			URL:    "http://localhost:80/v1/users/follows/communities/remove",
			Method: "POST",
			BodyJSON: map[string]interface{}{
				"user_id":      userID,
				"community_id": communityID,
			},
			ExpectedStatus: http.StatusOK,
		}

		c.Convey("if user does not exist", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, false, nil).Once()

			apitest.ExpectedStatus = http.StatusNotFound
			apitest.Expected = map[string]interface{}{
				"status": float64(http.StatusNotFound),
				"error":  api.ErrCodeRequestingUserNotFound,
			}
			testutils.RunTest(t, s, apitest)
			backenderMock.AssertExpectations(t)
		})

		c.Convey("if user exists", func() {
			backenderMock.On("GetSiteUser", mock.Anything, userID).
				Return(backend.SiteUser{}, true, nil).Once()

			c.Convey("if community is tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{
						TOSBanned: true,
					}, true, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityTOSBanned,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community does not exist", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, false, nil).Once()

				apitest.ExpectedStatus = http.StatusNotFound
				apitest.Expected = map[string]interface{}{
					"status": float64(http.StatusNotFound),
					"error":  api.ErrCodeCommunityIDNotFound,
				}
				testutils.RunTest(t, s, apitest)
				backenderMock.AssertExpectations(t)
			})

			c.Convey("if community exists and is not tos banned", func() {
				backenderMock.On("GetCommunity", mock.Anything, communityID).
					Return(backend.Community{}, true, nil).Once()

				c.Convey("with noop error", func() {
					backenderMock.On("RemoveUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(backend.ErrUserNotFollowingCommunity).Once()

					apitest.ExpectedStatus = http.StatusOK
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("with no error", func() {
					backenderMock.On("RemoveUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(nil).Once()

					apitest.ExpectedStatus = http.StatusOK
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})

				c.Convey("with not noop error", func() {
					backenderMock.On("RemoveUserFollowedCommunity", mock.Anything, userID, communityID).
						Return(errors.New("error")).Once()

					apitest.ExpectedStatus = http.StatusInternalServerError
					testutils.RunTest(t, s, apitest)
					backenderMock.AssertExpectations(t)
				})
			})
		})
	})
}
