package nuclear_waste

import (
	"fmt"
	"net/url"
	"strings"
	"time"

	"context"

	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "nuclear_waste"
)

type Client interface {
	// GetPremiumStatus returns a user's Twitch Turbo and Twitch Prime status.
	GetPremiumStatuses(ctx context.Context, userID string, ro *twitchclient.ReqOpts) (*PremiumStatuses, error)
	// Subscription gets the subscription benefits for `userID` as they relate to `channelID`'s subscription product.
	// Returns nil if `userID` does not have an active subscription benefit for `channelID`'s product.
	Subscription(ctx context.Context, userID, channelID string, ro *twitchclient.ReqOpts) (*Subscription, error)
	User(ctx context.Context, id string, reqOpts *twitchclient.ReqOpts) (*User, error)
	UserEmotes(ctx context.Context, userID string, ro *twitchclient.ReqOpts) (*EmoteCollection, error)
	// SDK Activation
	ActivateSDKLicense(ctx context.Context, clientID, channelID string, metaGame *string, ro *twitchclient.ReqOpts) error
	GetChannelFeedEnabled(ctx context.Context, channelID string, ro *twitchclient.ReqOpts) (*ChannelFeedEnabledResponse, error)
	SetChannelFeedEnabled(ctx context.Context, channelID string, enabled bool, ro *twitchclient.ReqOpts) error
}

type client struct {
	twitchclient.Client
}

type User struct {
	EmailNotifications bool `json:"email_notifications"`
	PushNotifications  bool `json:"push_notifications"`
	TwitterConnected   bool `json:"twitter_connected"`
}

type SubscriptionEmote struct {
	State    string `json:"state"`
	ImageURL string `json:"image_url"`
	Token    string `json:"token"`
}

// SubscriptionFeatures ...
type SubscriptionFeatures struct {
	BitrateAccess     []interface{} `json:"bitrate_access"`
	BaseEmoticonSetID int           `json:"base_emoticon_set_id"`
	EmoticonSetIDs    []int         `json:"emoticon_set_ids"`
	Tier              string        `json:"tier"`
}

// SubscriptionProduct ...
type SubscriptionProduct struct {
	ID             string               `json:"id"`
	Name           string               `json:"name"`
	Price          string               `json:"price"`
	Features       SubscriptionFeatures `json:"features"`
	Emotes         []SubscriptionEmote  `json:"emotes"`
	IntervalNumber int                  `json:"interval_number"`
	PartnerLogin   *string              `json:"partner_login"`
	OwnerName      *string              `json:"owner_name"`
	Period         string               `json:"period"`
	ShortName      string               `json:"short_name"`
	IsRecurring    bool                 `json:"recurring"`
	TicketType     string               `json:"ticket_type"`
}

type PurchaseProfile struct {
	ID              string    `json:"id"`
	IsExpired       bool      `json:"is_expired"`
	IsRefundable    bool      `json:"is_refundable"`
	MonthsTenured   int       `json:"number_months_tenured"`
	PaidOn          time.Time `json:"paid_on"`
	PaymentProvider string    `json:"payment_provider"`
	WillRenew       bool      `json:"will_renew"`
}

// Subscription benefits a user is entitled to after purchasing a SubscriptionProduct.
type Subscription struct {
	ID              int                 `json:"id"`
	AccessStart     *time.Time          `json:"access_start"`
	AccessEnd       time.Time           `json:"access_end"`
	IsExpired       bool                `json:"is_expired"`
	Product         SubscriptionProduct `json:"product"`
	PurchaseProfile PurchaseProfile     `json:"purchase_profile"`
}

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) User(ctx context.Context, id string, opts *twitchclient.ReqOpts) (*User, error) {
	query := url.Values{
		"id": {id},
	}

	req, err := c.NewRequest("GET", "/api/internal/nuclear_waste/user?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedOpts := twitchclient.MergeReqOpts(opts, twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.user",
		StatSampleRate: defaultStatSampleRate,
	})

	var data *User
	_, err = c.DoJSON(ctx, &data, req, combinedOpts)
	return data, err
}

// EmoteCollection contains a mapping of EmoteSetID to a list of Emotes in the set.
type EmoteCollection struct {
	Sets map[string][]Emote `json:"emoticon_sets"`
}

// Emote is the bare minimum representation of an Emote.
type Emote struct {
	ID    int    `json:"id"`
	Token string `json:"code"`
}

// UserEmotes returns the emotes which a user is entitled to use on the site.
func (c *client) UserEmotes(ctx context.Context, userID string, ro *twitchclient.ReqOpts) (*EmoteCollection, error) {
	query := url.Values{"id": {userID}}
	req, err := c.NewRequest("GET", "/api/internal/nuclear_waste/user_emotes?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.user_emotes",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)

	collection := &EmoteCollection{}
	_, err = c.DoJSON(ctx, collection, req, opts)
	if err != nil {
		return nil, err
	}

	return collection, nil
}

// PremiumStatuses has information on user membership to Twitch Turbo and Twitch Prime.
type PremiumStatuses struct {
	// Whether the user has Twitch Turbo.
	HasTurbo bool `json:"has_turbo"`
	// Whether the user has Twitch Prime.
	HasPrime bool `json:"has_prime"`
}

func (c *client) GetPremiumStatuses(ctx context.Context, userID string, ro *twitchclient.ReqOpts) (*PremiumStatuses, error) {
	query := url.Values{"id": {userID}}
	req, err := c.NewRequest("GET", "/api/internal/nuclear_waste/premium_statuses?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.get_premium_status",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)

	resp := &PremiumStatuses{}
	_, err = c.DoJSON(ctx, resp, req, opts)
	if err != nil {
		return nil, err
	}

	return resp, nil
}

func (c *client) Subscription(ctx context.Context, userID, channelID string, ro *twitchclient.ReqOpts) (*Subscription, error) {
	query := url.Values{"uid": {userID}, "cid": {channelID}}
	req, err := c.NewRequest("GET", "/api/internal/nuclear_waste/user_channel_sub?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.subscription",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)

	resp := &Subscription{}
	_, err = c.DoJSON(ctx, resp, req, opts)
	if err != nil {
		return nil, err
	}

	return resp, nil
}

func (c *client) ActivateSDKLicense(ctx context.Context, clientID, channelID string, metaGame *string, ro *twitchclient.ReqOpts) error {
	form := url.Values{"id": {channelID}, "client_id": {clientID}, "meta_game": {*metaGame}}
	req, err := c.NewRequest("POST", "/api/internal/nuclear_waste/licensing", strings.NewReader(form.Encode()))
	if err != nil {
		return err
	}

	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	req.PostForm = form

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.licensing",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)

	_, err = c.DoNoContent(ctx, req, opts)
	return err
}

type ChannelFeedEnabledResponse struct {
	ChannelFeedEnabled bool `json:"channel_feed_enabled"`
}

func (c *client) GetChannelFeedEnabled(ctx context.Context, channelID string, ro *twitchclient.ReqOpts) (*ChannelFeedEnabledResponse, error) {
	var resp ChannelFeedEnabledResponse
	url := fmt.Sprintf("/api/internal/nuclear_waste/get_channel_feed?id=%s", channelID)
	req, err := c.NewRequest("GET", url, nil)
	if err != nil {
		return nil, err
	}

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.get_channel_feed_enabled",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)
	_, err = c.DoJSON(ctx, &resp, req, opts)
	if err != nil {
		return nil, err
	}
	return &resp, nil
}

func (c *client) SetChannelFeedEnabled(ctx context.Context, channelID string, enabled bool, ro *twitchclient.ReqOpts) error {
	enabledString := "false"
	if enabled {
		enabledString = "true"
	}
	form := url.Values{"id": {channelID}, "channel_feed_enabled": {enabledString}}
	req, err := c.NewRequest("POST", "/api/internal/nuclear_waste/channel_feed", strings.NewReader(form.Encode()))
	if err != nil {
		return err
	}

	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	req.PostForm = form

	defaults := twitchclient.ReqOpts{
		StatName:       "service.nuclear_waste.set_channel_feed_enabled",
		StatSampleRate: defaultStatSampleRate,
	}
	opts := twitchclient.MergeReqOpts(ro, defaults)

	_, err = c.DoNoContent(ctx, req, opts)
	return err
}
