package feedsettings

import (
	"net/url"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"golang.org/x/net/context"
)

// Client is used by third parties to connect and interact with feed-settings
type Client struct {
	RequestDoer    clients.RequestDoer
	NewHTTPRequest clients.NewHTTPRequest
	Config         *Config
}

// Config configures feed-settings http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("feed-settings.http_endpoint", "").Get
	return nil
}

// Settings contains user settings
type Settings struct {
	Entity                string    `json:"entity"` // User ID or feed ID
	CreatedAt             time.Time `json:"created_at"`
	UpdatedAt             time.Time `json:"updated_at"`
	SubsCanComment        bool      `json:"subs_can_comment"`
	FriendsCanComment     bool      `json:"friends_can_comment"`
	FollowersCanComment   bool      `json:"followers_can_comment"`
	UserDisabledComments  bool      `json:"user_disabled_comments"`
	AdminDisabledComments bool      `json:"admin_disabled_comments"`
	ChannelFeedEnabled    bool      `json:"channel_feed_enabled"`
}

// UpdateSettingsOptions specifies the optional parameters of an UpdateSettings operation
type UpdateSettingsOptions struct {
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	FollowersCanComment   *bool `json:"followers_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
	ChannelFeedEnabled    *bool `json:"channel_feed_enabled,omitempty"`
}

// GetSettings returns the post for the given ID, or nil
func (c *Client) GetSettings(ctx context.Context, entity string) (*Settings, error) {
	path := "/v1/settings/" + entity
	var ret Settings
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// UpdateSettings creates a post with the given content
func (c *Client) UpdateSettings(ctx context.Context, entity string, params *UpdateSettingsOptions) (*Settings, error) {
	path := "/v1/settings/" + entity
	var ret Settings
	if err := c.http(ctx, "POST", path, nil, params, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

func (c *Client) http(ctx context.Context, method string, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return clients.DoHTTP(ctx, c.RequestDoer, method, c.Config.Host()+path, queryParams, body, into, c.NewHTTPRequest)
}
