// +build integration

package feedsettings_test

import (
	"testing"
	"time"

	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/service-common"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func timestamp() string {
	return time.Now().Format(time.RFC3339Nano)
}

func TestClientIntegration_Settings(t *testing.T) {
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		t.Log("using client addr ", c.Config.Host())

		Convey("Should get settings", func() {
			entity := "clientintegration:" + timestamp()
			settings, err := c.GetSettings(ctx, entity)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldEqual, entity)
		})

		Convey("Should get default settings for a nonexistent entity", func() {
			entity := "clientintegration:bogus:" + timestamp()
			settings, err := c.GetSettings(ctx, entity)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldEqual, entity)
			// default settings
			So(settings.SubsCanComment, ShouldBeTrue)
			So(settings.FriendsCanComment, ShouldBeTrue)
			So(settings.FollowersCanComment, ShouldBeTrue)
			So(settings.UserDisabledComments, ShouldBeFalse)
			So(settings.AdminDisabledComments, ShouldBeFalse)
			So(settings.ChannelFeedEnabled, ShouldBeTrue)
		})

		Convey("Should update settings", func() {
			entity := "clientintegration:" + timestamp()

			settings, err := c.GetSettings(ctx, entity)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldEqual, entity)

			updates := &feedsettings.UpdateSettingsOptions{
				AdminDisabledComments: boolPtr(!settings.AdminDisabledComments),
				SubsCanComment:        boolPtr(!settings.SubsCanComment),
				FriendsCanComment:     boolPtr(!settings.FriendsCanComment),
				FollowersCanComment:   boolPtr(!settings.FollowersCanComment),
				UserDisabledComments:  boolPtr(!settings.UserDisabledComments),
				ChannelFeedEnabled:    boolPtr(!settings.ChannelFeedEnabled),
			}
			updated, err := c.UpdateSettings(ctx, entity, updates)
			So(err, ShouldBeNil)
			So(updated.Entity, ShouldEqual, entity)
			So(updated.AdminDisabledComments, ShouldEqual, *updates.AdminDisabledComments)
			So(updated.SubsCanComment, ShouldEqual, *updates.SubsCanComment)
			So(updated.FriendsCanComment, ShouldEqual, *updates.FriendsCanComment)
			So(updated.FollowersCanComment, ShouldEqual, *updates.FollowersCanComment)
			So(updated.UserDisabledComments, ShouldEqual, *updates.UserDisabledComments)
			So(updated.ChannelFeedEnabled, ShouldEqual, *updates.ChannelFeedEnabled)

			settings2, err := c.GetSettings(ctx, entity)
			So(err, ShouldBeNil)
			So(settings2, ShouldResemble, updated)
		})
	})
}

func boolPtr(b bool) *bool {
	return &b
}

type TestSetup struct {
	context.Context
	*feedsettings.Client
	cancelFunc func()
	Common     service_common.ConfigCommon
}

func newSetup() *TestSetup {
	return &TestSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	cfg := &feedsettings.Config{}
	if err := cfg.Load(t.Common.Config); err != nil {
		return err
	}
	t.Client = &feedsettings.Client{
		Config: cfg,
	}
	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}
