package masonry

import (
	"fmt"
	"net/url"
	"strconv"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"golang.org/x/net/context"
)

// Config configures masonry http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("masonry.http_endpoint", "").Get
	return nil
}

// A Feed is a list of activities
type Feed struct {
	ID       string        `json:"id"`
	Cursor   string        `json:"cursor"`
	Items    []Activity    `json:"items"`
	Tracking *FeedTracking `json:"tracking,omitempty"`
}

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity entity.Entity `json:"entity"`
	Verb   verb.Verb     `json:"verb"`
	Actor  entity.Entity `json:"actor"`

	RecGenerationID    string                 `json:"rec_generation_id,omitempty"`
	RecGenerationIndex *int                   `json:"rec_generation_index,omitempty"`
	RelevanceReason    *RelevanceReason       `json:"relevance_reason,omitempty"`
	Metadata           *feeddataflow.Metadata `json:"metadata,omitempty"`
}

// RelevanceReason describes why an Activity is being added to a feed.
type RelevanceReason struct {
	// Kind is a keyword that identifies the reason.
	Kind string `json:"kind"`

	// Source identifies the service or component that created this reason.
	Source string `json:"source"`

	// EntityID identifies who or what the reason relates to.  For example, the EntityID for a followed_channel
	// reason would be the channel ID of the channel that was followed.
	EntityID string `json:"entity_id,omitempty"`
}

const (
	// ReasonSourceRecommendations identifies relevance reasons that came from the recommendations service.
	ReasonSourceRecommendations = "recs"
)

// FeedTracking holds feed level tracking information
type FeedTracking struct {
	// BatchID The batch ID is a tracking parameter meant to group events associated with a user's "session"
	// with a feed.  A fresh reload of the feed gets a new batch ID, while paging deeper into the feed will re-use
	// the existing batch ID.
	BatchID string `json:"batch_id,omitempty"`
}

type errResponse struct {
	StatusCode int    `json:"-"`
	Message    string `json:"message"`
}

func (e errResponse) Error() string {
	return fmt.Sprintf("%d: %s", e.StatusCode, e.Message)
}

// Client does normal HTTP requests to get information from masonry
type Client struct {
	RequestDoer    clients.RequestDoer
	NewHTTPRequest clients.NewHTTPRequest
	Config         *Config
}

func (c *Client) http(ctx context.Context, method, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return clients.DoHTTP(ctx, c.RequestDoer, method, c.Config.Host()+path, queryParams, body, into, c.NewHTTPRequest)
}

// GetFeedOptions defines optional params when getting a feed
type GetFeedOptions struct {
	Limit    int
	Cursor   string
	DeviceID string
	Language string
}

// GetFeed returns a paginated feed
func (c *Client) GetFeed(ctx context.Context, feedID string, options *GetFeedOptions) (*Feed, error) {
	path := "/v1/feeds/" + feedID
	query := url.Values{}
	if options != nil {
		if options.Limit != 0 {
			query.Add("limit", strconv.Itoa(options.Limit))
		}
		if options.Cursor != "" {
			query.Add("cursor", options.Cursor)
		}
		if options.DeviceID != "" {
			query.Add("device_id", options.DeviceID)
		}
		if options.Language != "" {
			query.Add("language", options.Language)
		}
	}
	var ret Feed
	if err := c.http(ctx, "GET", path, query, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}
