// +build integration

package masonry

import (
	"testing"
	"time"

	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/service/sqs"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

const testrecfeedUserID = "159182595"

func TestIntegration_QueueClient(t *testing.T) {
	Convey("With a client", t, func() {
		s := service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "int_tests",
			BaseDirectory: "../",
		}
		So(s.Setup(), ShouldBeNil)
		session, awsConf := service_common.CreateAWSSession(s.Config)
		sqsClient := sqs.New(session, awsConf...)
		lqi := sqs.ListQueuesInput{}
		lqOut, err := sqsClient.ListQueues(&lqi)
		So(err, ShouldBeNil)
		So(len(lqOut.QueueUrls), ShouldBeGreaterThan, 0)
	})
}

func TestIntegration_RecommendationFeed(t *testing.T) {
	Convey("masonry client", t, func() {

		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})

		ctx := setup.Context
		masonryClient := setup.masonryClient

		Convey("getFeed", func() {
			Convey("when given a recommendation feed", func() {

				feed, err := masonryClient.GetFeed(ctx, "r:"+testrecfeedUserID, &GetFeedOptions{
					Limit: 2,
				})

				Convey("succeeds", func() {
					So(err, ShouldBeNil)
					So(feed, ShouldNotBeNil)
					So(feed.Cursor, ShouldNotBeBlank)
					So(feed.Tracking, ShouldNotBeNil)
					So(feed.Tracking.BatchID, ShouldNotBeBlank)

					So(feed.Items, ShouldHaveLength, 2)

					item := feed.Items[0]
					So(item, ShouldNotBeNil)
					So(item.Entity.ID(), ShouldNotBeEmpty)
					So(item.RelevanceReason, ShouldNotBeNil)
					So(item.RelevanceReason.Source, ShouldEqual, ReasonSourceRecommendations)
					So(item.RelevanceReason.Kind, ShouldNotEqual, "")
				})
			})
		})
	})
}

func newSetup() *TestSetup {
	return &TestSetup{
		CommonConfig: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

type TestSetup struct {
	Context       context.Context
	masonryClient Client
	cancelFunc    func()
	CommonConfig  service_common.ConfigCommon
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)

	if err := t.CommonConfig.Setup(); err != nil {
		return err
	}

	masonryConfig := &Config{}
	if err := masonryConfig.Load(t.CommonConfig.Config); err != nil {
		return err
	}

	t.masonryClient = Client{
		Config: masonryConfig,
	}

	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.CommonConfig.Close()
}
