// +build integration

package shine_test

import (
	"testing"
	"time"

	"code.justin.tv/feeds/clients/shine"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestIntegrationShineClient(t *testing.T) {
	Convey("With a client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client

		Convey("Should return an embed object, when given a clip URL", func() {
			testURL := "https://clips.twitch.tv/manvsgame/AmusedLarkPMSTwin"
			autoplay := false
			embed, err := c.GetEmbed(ctx, testURL, &shine.GetEmbedOptions{Autoplay: &autoplay}, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.ProviderName, ShouldEqual, "Twitch")
		})

		Convey("Should return a nil object", func() {
			testURL := "https://test.com"
			autoplay := false
			embed, err := c.GetEmbed(ctx, testURL, &shine.GetEmbedOptions{Autoplay: &autoplay}, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldBeNil)
		})

		Convey("Should return an embed object, when given a valid clip entity", func() {
			clipEntity := entity.New(entity.NamespaceClip, "SneakyUnusualBeefCclamChamp")
			autoplay := false
			embed, err := c.GetEmbedForEntity(ctx, clipEntity, &shine.GetEmbedOptions{Autoplay: &autoplay}, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
			So(embed.ProviderName, ShouldEqual, "Twitch")
		})

		Convey("Should return an EntitiesForURLs object, when given multiple URLs", func() {
			clipURL := "https://clips.twitch.tv/TubularSuccessfulSardineSoonerLater"
			streamURL := "https://www.twitch.tv/broadcasts/25464189456/channel/27446517"

			bulkURLs := []string{clipURL, streamURL}
			entities, err := c.GetEntitiesForURLs(ctx, bulkURLs, nil)
			So(err, ShouldBeNil)
			So(entities, ShouldNotBeNil)
			So(len(bulkURLs), ShouldEqual, len(entities.Entities))
		})

		Convey("Should return an error when GetEntitiesForURLs is given zero URLs", func() {
			entities, err := c.GetEntitiesForURLs(ctx, []string{}, nil)
			So(err, ShouldNotBeNil)
			So(entities, ShouldBeNil)
		})

		Convey("Should return en OembedsForURLs obejct when given multiple URLs", func() {
			youtubeURL := "https://www.youtube.com/watch?v=EvxS_bJ0yOU"
			vimeoURL := "https://vimeo.com/227690432"

			bulkURLs := []string{youtubeURL, vimeoURL}
			oembeds, err := c.GetOembedsForURLs(ctx, bulkURLs, nil, nil)
			So(err, ShouldBeNil)
			So(oembeds, ShouldNotBeNil)
			So(len(bulkURLs), ShouldEqual, len(oembeds.Oembeds))
		})

		Convey("Should return an error when GetOembedsForURLs is given zero URLs", func() {
			_, err := c.GetOembedsForURLs(ctx, []string{}, nil, nil)
			So(err, ShouldNotBeNil)
		})
	})
}

func newSetup() *TestSetup {
	return &TestSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "int_tests",
			BaseDirectory: "../",
		},
	}
}

// Config configures edge http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("shine.http_endpoint", "").Get
	return nil
}

type TestSetup struct {
	context.Context
	shine.Client
	cancelFunc func()
	Common     service_common.ConfigCommon
	Config     *Config
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	t.Config = &Config{}
	if err := t.Config.Load(t.Common.Config); err != nil {
		return err
	}

	var err error
	t.Client, err = shine.NewClient(twitchclient.ClientConf{
		Host: t.Config.Host(),
	})
	if err != nil {
		return err
	}
	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}
