package sfxstatsd

import (
	"context"
	"expvar"

	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/feeds/metrics/sfx/sfxresink"
	"code.justin.tv/feeds/metrics/statsdim"
	"github.com/signalfx/golib/sfxclient"
)

type SetupConfig struct {
	CounterMetricRules []*statsdim.ConfigurableDelimiterMetricRule `nilcheck:"ignore"`
	TimingMetricRules  []*statsdim.ConfigurableDelimiterMetricRule `nilcheck:"ignore"`
	SetMetricRules     []*statsdim.ConfigurableDelimiterMetricRule `nilcheck:"ignore"`
	GaugeMetricRules   []*statsdim.ConfigurableDelimiterMetricRule `nilcheck:"ignore"`

	AuthToken      string
	ReportingDelay time.Duration
	DefaultDimensions map[string]string
}

type SetupResult struct {
	Scheduler        *sfxclient.Scheduler `nilcheck:"nodepth"`
	StatsdTranslator *statsdim.StatsdTranslator
	Statter          Statter
}

func AWSUniqueID(ctx context.Context, c *http.Client) (string, error) {
	if c == nil {
		c = http.DefaultClient
	}
	req, err := http.NewRequest("GET", "http://169.254.169.254/latest/dynamic/instance-identity/document", nil)
	if err != nil {
		return "", err
	}
	req = req.WithContext(ctx)
	resp, err := c.Do(req)
	if err != nil {
		return "", err
	}
	if resp.StatusCode != 200 {
		return "", fmt.Errorf("invalid status code: %d", resp.StatusCode)
	}
	var decodeInto = struct {
		Region     string `json:"region"`
		InstanceId string `json:"instanceId"`
		AccountID  string `json:"accountId"`
	}{}
	if err := json.NewDecoder(resp.Body).Decode(&decodeInto); err != nil {
		return "", err
	}
	if decodeInto.Region == "" {
		return "", errors.New("empty Region from AWS endpoint")
	}
	if decodeInto.InstanceId == "" {
		return "", errors.New("empty InstanceId from AWS endpoint")
	}
	if decodeInto.AccountID == "" {
		return "", errors.New("empty AccountID from AWS endpoint")
	}
	return fmt.Sprintf("%s_%s_%s", decodeInto.InstanceId, decodeInto.Region, decodeInto.AccountID), nil
}

const AWSUniqueIdKey = "AWSUniqueId"

func copyDims(a map[string]string, b map[string]string) map[string]string {
	ret := make(map[string]string, len(a) + len(b))
	for k ,v := range a {
		ret[k] = v
	}
	for k ,v := range b {
		ret[k] = v
	}
	return ret
}

func DefaultSetup(c SetupConfig) (*SetupResult, error) {
	statsdMetricStore := &TranslatedMetricsStore{}
	statsdTranslator := statsdim.DefaultTranslator()
	statsdTranslator.CounterTranslator.MetricRules = append(statsdTranslator.CounterTranslator.MetricRules,
		c.CounterMetricRules...,
	)
	statsdTranslator.SetTranslator.MetricRules = append(statsdTranslator.SetTranslator.MetricRules,
		c.SetMetricRules...,
	)
	statsdTranslator.GaugeTranslator.MetricRules = append(statsdTranslator.GaugeTranslator.MetricRules,
		c.GaugeMetricRules...,
	)
	statsdTranslator.TimingTranslator.MetricRules = append(statsdTranslator.TimingTranslator.MetricRules,
		c.TimingMetricRules...,
	)
	if err := statsdTranslator.Setup(); err != nil {
		return nil, err
	}
	scheduler := sfxclient.NewScheduler()
	scheduler.Sink.(*sfxclient.HTTPSink).AuthToken = c.AuthToken
	scheduler.Sink = &sfxresink.Resink{
		Fallthrough: scheduler.Sink,
		RemovedDimensions: map[string]struct{}{
			"containerId": {},
		},
	}
	scheduler.AddCallback(statsdMetricStore)
	ctx, ctxCancel := context.WithTimeout(context.Background(), time.Millisecond*500)
	defer ctxCancel()
	defaultDims := copyDims(c.DefaultDimensions, nil)
	if awsID, err := AWSUniqueID(ctx, nil); err == nil {
		defaultDims[AWSUniqueIdKey] = awsID
	}
	scheduler.DefaultDimensions(defaultDims)

	if c.ReportingDelay.Nanoseconds() == 0 {
		scheduler.ReportingDelay(time.Second * 60)
	} else {
		scheduler.ReportingDelay(c.ReportingDelay)
	}

	return &SetupResult{
		Scheduler:        scheduler,
		StatsdTranslator: &statsdTranslator,
		Statter: Statter{
			StatSender: StatSender{
				Store:      statsdMetricStore,
				Translator: &statsdTranslator,
			},
		},
	}, nil
}

func SingletonSetupAndInstall(c SetupConfig) (*SetupResult, error) {
	ret, err := DefaultSetup(c)
	if err != nil {
		return nil, err
	}
	go ret.Scheduler.Schedule(context.Background())
	expvar.Publish("datapoints", ret.Scheduler.Var())
	return ret, nil
}

func MustSingletonSetupAndInstall(c SetupConfig) *SetupResult {
	if ret, err := SingletonSetupAndInstall(c); err != nil {
		panic(err)
	} else {
		return ret
	}
}
