package statsdim

import (
	"fmt"
	"sync"
)

// StatsdTranslator can translate statsd metrics into dimensional metrics
type StatsdTranslator struct {
	CounterTranslator   ConfigurableDelimiterMetricDeconstructor `nilcheck:"ignore"`
	GaugeTranslator     ConfigurableDelimiterMetricDeconstructor `nilcheck:"ignore"`
	TimingTranslator    ConfigurableDelimiterMetricDeconstructor `nilcheck:"ignore"`
	SetTranslator       ConfigurableDelimiterMetricDeconstructor `nilcheck:"ignore"`
	Prefix              string
	LogMisses           bool
	LogCallback         func(metricName string, metricType string, err error) `nilcheck:"ignore"`
	StrictMode          bool
	alreadyLoggedKeysMu sync.Mutex
	alreadyLoggedKeys   map[string]struct{} `nilcheck:"ignore"`
}

// DefaultTranslator creates a translator that includes matches for common twitch patterns
func DefaultTranslator() StatsdTranslator {
	return StatsdTranslator{
		LogMisses: true,
		GaugeTranslator: ConfigurableDelimiterMetricDeconstructor{
			MetricRules: []*ConfigurableDelimiterMetricRule{
				{
					MetricPath:    "go.*",
					DimensionsMap: "lang.%",
				},
				{
					MetricPath:    "go.*.*",
					DimensionsMap: "lang.%.%",
				},
				{
					MetricPath:    "go.process.uptime.ns",
					DimensionsMap: "lang.%.%.-",
				},
			},
		},
		TimingTranslator: ConfigurableDelimiterMetricDeconstructor{
			MetricRules: []*ConfigurableDelimiterMetricRule{
				{
					MetricPath:    "go.gc.*",
					DimensionsMap: "lang.%.%",
				},
				{
					MetricPath:    "twirp.all_methods.response",
					DimensionsMap: "%.%.%",
					MetricName:    "timing",
				},
				{
					MetricPath:    "twirp.*.response",
					DimensionsMap: "%.method.%",
					MetricName:    "timing",
				},
				{
					MetricPath:    "twirp.status_codes.all_methods.*",
					DimensionsMap: "%.%.%.status_code",
					MetricName:    "timing",
				},
				{
					MetricPath:    "twirp.status_codes.*.*",
					DimensionsMap: "%.%.method.status_code",
					MetricName:    "timing",
				},
				{
					MetricPath:    "dns.*.*",
					DimensionsMap: "%.hostname.%",
				},
				{
					MetricPath:    "service.*.*.*",
					DimensionsMap: "-.external_service.method.http_code",
					MetricName:    "service_call_times",
				},
			},
		},
		CounterTranslator: ConfigurableDelimiterMetricDeconstructor{
			MetricRules: []*ConfigurableDelimiterMetricRule{
				{
					MetricPath:    "go.*",
					DimensionsMap: "lang.%",
				},
				{
					MetricPath:    "go.*.*",
					DimensionsMap: "lang.%.%",
				},
				{
					MetricPath:    "twirp.total.requests",
					DimensionsMap: "%.%.%",
					MetricName:    "count",
				},
				{
					MetricPath:    "twirp.total.responses",
					DimensionsMap: "%.%.%",
					MetricName:    "count",
				},
				{
					MetricPath:    "twirp.*.requests",
					DimensionsMap: "%.method.%",
					MetricName:    "count",
				},
				{
					MetricPath:    "twirp.*.responses",
					DimensionsMap: "%.method.%",
					MetricName:    "count",
				},
				{
					MetricPath:    "twirp.status_codes.total.*",
					DimensionsMap: "%.%.%.status_code",
					MetricName:    "count",
				},
				{
					MetricPath:    "twirp.status_codes.*.*",
					DimensionsMap: "%.%.method.status_code",
					MetricName:    "count",
				},
			},
		},
	}
}

func (s *StatsdTranslator) Setup() error {
	if err := DelimiterLoader(&s.GaugeTranslator); err != nil {
		return err
	}
	if err := DelimiterLoader(&s.TimingTranslator); err != nil {
		return err
	}
	if err := DelimiterLoader(&s.CounterTranslator); err != nil {
		return err
	}
	if err := DelimiterLoader(&s.SetTranslator); err != nil {
		return err
	}
	return nil
}

func (s *StatsdTranslator) checkErr(metric string, t string, err error) {
	if s.LogMisses && err != nil {
		s.alreadyLoggedKeysMu.Lock()
		defer s.alreadyLoggedKeysMu.Unlock()
		if _, exists := s.alreadyLoggedKeys[metric]; exists {
			return
		}
		if s.alreadyLoggedKeys == nil {
			s.alreadyLoggedKeys = make(map[string]struct{}, 24)
		}
		s.alreadyLoggedKeys[metric] = struct{}{}
		if s.LogCallback != nil {
			s.LogCallback(metric, t, err)
		} else {
			fmt.Printf("Missing metric %s for type %s: %s\n", metric, t, err)
		}
	}
}

func (m *StatsdTranslator) FullKey(k string) string {
	if m.Prefix == "" {
		return k
	}
	return m.Prefix + "." + k
}

func (m *StatsdTranslator) Inc(k string) (string, map[string]string, error) {
	return m.fromTranslator(k, &m.CounterTranslator, "counter")
}

func (m *StatsdTranslator) fromTranslator(k string, translator *ConfigurableDelimiterMetricDeconstructor, dType string) (string, map[string]string, error) {
	metricName, dims, err := translator.Parse(m.FullKey(k))
	if err != nil {
		m.checkErr(k, dType, err)
		if m.StrictMode {
			return "", nil, err
		}
		return fmt.Sprintf("untranslated.%s.%s", dType, k), nil, nil
	}
	return metricName, dims, nil
}

func (m *StatsdTranslator) Dec(k string) (string, map[string]string, error) {
	return m.Inc(k)
}

func (m *StatsdTranslator) Gauge(k string) (string, map[string]string, error) {
	return m.fromTranslator(k, &m.GaugeTranslator, "gauge")
}

func (m *StatsdTranslator) GaugeDelta(k string) (string, map[string]string, error) {
	return m.Gauge(k)
}

func (m *StatsdTranslator) Timing(k string) (string, map[string]string, error) {
	return m.fromTranslator(k, &m.TimingTranslator, "timing")
}

func (m *StatsdTranslator) TimingDuration(k string) (string, map[string]string, error) {
	return m.Timing(k)
}

func (m *StatsdTranslator) Set(k string) (string, map[string]string, error) {
	return m.fromTranslator(k, &m.SetTranslator, "set")
}

func (m *StatsdTranslator) SetInt(k string) (string, map[string]string, error) {
	return m.Set(k)
}

func (m *StatsdTranslator) SetPrefix(k string) {
	m.Prefix = k
}

func (m *StatsdTranslator) NewSubStatter(k string) *StatsdTranslator {
	return &StatsdTranslator{
		SetTranslator:     m.SetTranslator,
		TimingTranslator:  m.TimingTranslator,
		GaugeTranslator:   m.GaugeTranslator,
		CounterTranslator: m.CounterTranslator,
		Prefix:            m.FullKey(k),
		LogMisses:         m.LogMisses,
		LogCallback:       m.LogCallback,
		StrictMode: m.StrictMode,
	}
}
