package xray

import "context"

// Capture traces an arbitrary synchronous code segment
func (x *XRay) Capture(ctx context.Context, name string, fn func(context.Context) error) (err error) {
	ctx, seg := x.newSegment(ctx, name)
	defer func() { seg.close(err, x) }()

	defer func() {
		if p := recover(); p != nil {
			err = panicf("%v", p)
			panic(p)
		}
	}()

	return fn(ctx)
}

// CaptureAsync traces an arbitrary code segment within a goroutine.
// Use CaptureAsync instead of manually calling Capture within a goroutine
// to ensure the segment is flushed properly.
func (x *XRay) CaptureAsync(ctx context.Context, name string, fn func(context.Context) error) <-chan error {
	started := make(chan struct{})
	res := make(chan error, 1)
	go func() {
		err := x.Capture(ctx, name, func(ctx context.Context) error {
			close(started)
			return fn(ctx)
		})
		res <- err
	}()
	<-started
	return res
}
